<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\FirebaseService;
use App\Models\User;
use App\Models\Notification;

class NotificationController extends Controller
{
    public function send(Request $request)
    {
        $title = $request->title;
        $body = $request->message;
        $firebase = new FirebaseService();
        
        if ($request->user_id) {
            $user = User::find($request->user_id);
            if ($user && $user->fcm_token) {
                $firebase->sendPushNotification($user->fcm_token, $title, $body);
            }
        } else {
            $tokens = User::whereNotNull('fcm_token')->pluck('fcm_token')->toArray();
            foreach ($tokens as $token) {
                $firebase->sendPushNotification($token, $title, $body);
            }
        }
        //dd($request->all());

        return redirect()->back()->with('success', 'Notification sent!');
    }

    public function updateToken(Request $request)
    {
        // Validate the incoming request
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        // Get the authenticated user
        $user = $request->user();
        $user->fcm_token = $request->fcm_token;
        $user->save();

        // Respond with success message
        return response()->json([
            'status' => 'success',
            'user'=>$user,
            'receive_token'=>$request->fcm_token,
            'message' => 'FCM token updated successfully.',
        ], 200);
    }

    //////////////////
      /**
     * Get all notifications.
     */
    public function getNotifications()
    {
        return response()->json(Notification::all());
    }

    /**
     * Get a single notification by ID.
     */
    public function getNotificationById($id)
    {
        $notification = Notification::find($id);
        if (!$notification) {
            return response()->json(['message' => 'Notification not found'], 404);
        }
        return response()->json($notification);
    }

    /**
     * Create a new notification.
     */
    public function createNotification(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'message' => 'required|string',
            'user_id' => 'nullable|exists:users,id', // in case you want to send to a specific user
        ]);

        // First, try to send notification
        $title = $request->title;
        $body = $request->message;
        $firebase = new FirebaseService();

        try {
            if ($request->user_id) {
                $user = User::find($request->user_id);
                if ($user && $user->fcm_token) {
                    $firebase->sendPushNotification($user->fcm_token, $title, $body);
                } else {
                    return response()->json(['message' => 'User not found or no FCM token'], 404);
                }
            } else {
                $tokens = User::whereNotNull('fcm_token')->pluck('fcm_token')->toArray();
                if (empty($tokens)) {
                    return response()->json(['message' => 'No FCM tokens found'], 404);
                }
                foreach ($tokens as $token) {
                    $firebase->sendPushNotification($token, $title, $body);
                }
            }
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to send notification', 'error' => $e->getMessage()], 500);
        }

        // If sending is successful, now save to database
        $notification = Notification::create([
            'title' => $request->title,
            'message' => $request->message,
        ]);

        return response()->json(['message' => 'Notification sent and saved successfully', 'notification' => $notification], 201);
    }


    /**
     * Update a notification.
     */
    public function updateNotification(Request $request, $id)
    {
        $notification = Notification::find($id);
        if (!$notification) {
            return response()->json(['message' => 'Notification not found'], 404);
        }

        $request->validate([
            'title' => 'sometimes|required|string',
            'message' => 'sometimes|required|string',
        ]);

        $notification->update([
            'title' => $request->title ?? $notification->title,
            'message' => $request->message ?? $notification->message,
        ]);

        return response()->json(['message' => 'Notification updated successfully', 'notification' => $notification]);
    }

    /**
     * Delete a notification.
     */
    public function deleteNotification($id)
    {
        $notification = Notification::find($id);
        if (!$notification) {
            return response()->json(['message' => 'Notification not found'], 404);
        }

        $notification->delete();

        return response()->json(['message' => 'Notification deleted successfully']);
    }
}
