<?php

namespace App\Http\Controllers;

use App\Models\Popup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;


class PopupController extends Controller
{

    public function getPopups() {
        return response()->json(Popup::all());
    }

    public function getPopupsApp()
    {
        $now = Carbon::now();

        $popup = Popup::whereDate('expire_date', '>=', $now)
        ->whereRaw('DATE_ADD(created_at, INTERVAL duration DAY) >= ?', [$now])
        ->where('status','active')
        ->orderBy('created_at', 'desc')
        ->first();

        return response()->json($popup);
    }

    /**
     * Show a specific popup.
     */
    public function getPopupById($id)
    {
        $popup = Popup::find($id);
        if (!$popup) {
            return response()->json(['message' => 'Popup not found'], 404);
        }
        return response()->json($popup);
    }

    /**
     * Create a new popup.
     */
    public function createPopup(Request $request)
    {
        $request->validate([
            'duration' => 'required|integer',
            'expire_date' => 'required|date',
            'popup_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Handle image upload
        if ($request->hasFile('popup_image')) {
            $filename = time() . '_' . $request->file('popup_image')->getClientOriginalName();
            $request->file('popup_image')->move(public_path('uploads/popup'), $filename);
            $imagePath = 'uploads/popup/' . $filename;
        } else {
            return response()->json(['message' => 'Image upload failed'], 500);
        }

        $popup = Popup::create([
            'duration' => $request->duration,
            'expire_date' => $request->expire_date,
            'popup_image' => $imagePath,
        ]);

        return response()->json(['message' => 'Popup created successfully', 'popup' => $popup], 201);
    }

    /**
     * Update popup.
     */
    public function updatePopup(Request $request, $id)
    {
        $popup = Popup::find($id);
        if (!$popup) {
            return response()->json(['message' => 'Popup not found'], 404);
        }

        $request->validate([
            'duration' => 'integer',
            'expire_date' => 'date',
            'popup_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Handle image upload
        if ($request->hasFile('popup_image')) {
            if ($popup->popup_image && file_exists(public_path($popup->popup_image))) {
                unlink(public_path($popup->popup_image));
            }

            $filename = time() . '_' . $request->file('popup_image')->getClientOriginalName();
            $request->file('popup_image')->move(public_path('uploads/popup'), $filename);
            $imagePath = 'uploads/popup/' . $filename;
        } else {
            $imagePath = $popup->popup_image;
        }

        $popup->update([
            'duration' => $request->duration ?? $popup->duration,
            'expire_date' => $request->expire_date ?? $popup->expire_date,
            'popup_image' => $imagePath,
        ]);

        return response()->json(['message' => 'Popup updated successfully', 'popup' => $popup]);
    }

    /**
     * Delete popup.
     */
    public function deletePopup($id)
    {
        $popup = Popup::find($id);
        if (!$popup) {
            return response()->json(['message' => 'Popup not found'], 404);
        }

        // Delete image
        if ($popup->popup_image && file_exists(public_path($popup->popup_image))) {
            unlink(public_path($popup->popup_image));
        }

        $popup->delete();

        return response()->json(['message' => 'Popup deleted successfully']);
    }

    public function changeStatusPopup($id)
    {
        $popup = Popup::find($id);
        if (!$popup) {
            return response()->json(['message' => 'Popup not found'], 404);
        }

        // Toggle status
        $popup->status = $popup->status === 'active' ? 'in-active' : 'active';
        $popup->save();

        return response()->json(['message' => 'Popup status updated successfully']);
    }


}
