<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Challenge;
use App\Models\ChallengeParticipant;
use App\Services\FirebaseService;
use App\Models\User;
use App\Models\Notification;

class ChallengeController extends Controller
{
    public function getChallenge($code) {
        $challenge = Challenge::where('challenge_code', $code)->first();
        return $challenge ? response()->json($challenge) : response()->json(['message' => 'Challenge not found'], 404);
    }

    public function createChallenge(Request $request) {
        // return response()->json([
        //     "req" => $request->all()
        // ]);

        $request->validate([
            'game_name' => 'required|string',
            'challenge_data' => 'required|string',
            'challenge_time' => 'required|string',
            'challenging_row' => 'nullable|integer|min:1|max:8'
        ]);
    
        do {
            $challengeCode = mt_rand(10000000, 99999999); 
        } while (Challenge::where('challenge_code', $challengeCode)->exists());
        
        $user = Auth::user();
        $challenge = Challenge::create([
            'creator_id' => $user->id,
            'game_name' => $request->game_name,
            'challenge_data' => $request->challenge_data,
            'challenging_row' => $request->challenging_row,
            'challenge_time' => $request->challenge_time,
            'challenge_code' => "$challengeCode",
            'status' => 'pending',
        ]);
    
        return response()->json(['challenge_id' => $challenge->challenge_code, 'message' => 'Challenge created successfully']);
    }

    public function updateChallengeStatus(Request $request) {
        $request->validate([
            'status' => 'required',
            'challenge_code' => 'required',
        ]);
    
        $challenge = Challenge::where('challenge_code', $request->challenge_code)->first();
    
        if (!$challenge) {
            return response()->json(['message' => 'Challenge not found'], 404);
        }
    
        $challenge->update(['status' => $request->status]);
    
        return response()->json([
            'message' => 'Challenge status updated successfully',
            'challenge' => $challenge
        ]);
    }

    
    
    // 1️⃣ JOIN A CHALLENGE
    public function joinChallenge(Request $request) {
        $request->validate([
            'challenge_code' => 'required|string|exists:challenges,challenge_code'
        ]);

        $user = Auth::user();
        $challenge = Challenge::where('challenge_code', $request->challenge_code)->first();

        // Check if the user already joined
        if (ChallengeParticipant::where('challenge_id', $challenge->id)->where('user_id', $user->id)->exists()) {
            return response()->json(['message' => 'You have already joined this challenge.'], 400);
        }

        // Add user to the challenge participants table
        $participant = ChallengeParticipant::create([
            'challenge_id' => $challenge->id,
            'user_id' => $user->id,
            'status' => 'pending'
        ]);

        return response()->json([
            'message' => 'Challenge joined successfully!',
            'challenge' => $challenge,
            'participant' => $participant
        ]);
    }

    // 2️⃣ UPDATE CHALLENGE RESULT (WIN/LOSS)
    public function updateChallengeResult(Request $request) {
        $request->validate([
            'challenge_code' => 'required|string|exists:challenges,challenge_code',
            'status' => 'required|in:won,lose'
        ]);
    
        $user = Auth::user();
        $challenge = Challenge::where('challenge_code', $request->challenge_code)->first();
    
        $participant = ChallengeParticipant::where('challenge_id', $challenge->id)
                                           ->where('user_id', $user->id)
                                           ->first();
    
        if (!$participant) {
            return response()->json(['message' => 'You have not joined this challenge.'], 404);
        }
    
        $participant->update(['status' => $request->status]);
    
        // 🔔 Notification logic (same as in updateChallengeStatus)
        if (in_array($request->status, ['won', 'lose']) && $challenge->creator_id) {
            $senderName = $user?->fullname ?? 'Someone';
            $creator = User::find($challenge->creator_id);
    
            if ($creator) {
                $firebase = new FirebaseService();
    
                $title = 'Challenge Result';
                $message = $request->status === 'won'
                    ? "$senderName has won your challenge! 🎉"
                    : "$senderName has lost your challenge. Better luck next time!";
    
                // Send FCM push notification
                if ($creator->fcm_token) {
                    $firebase->sendPushNotification($creator->fcm_token, $title, $message);
                }
    
                // Store notification in DB
                Notification::create([
                    'title' => $title,
                    'message' => $message,
                    'user_id' => $creator->id,
                ]);
            }
        }
    
        return response()->json([
            'message' => 'Challenge result updated successfully!',
            'participant' => $participant
        ]);
    }

    
}
