<?php
namespace App\Http\Controllers;

use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;


class BannerController extends Controller {
    /**
     * Get all banners.
     */
    public function getBanners() {
        return response()->json(Banner::all());
    }

    public function getBannersApp()
    {
        $now = Carbon::now();
    
        $banners = Banner::whereDate('expire_date', '>=', $now)
            ->whereRaw('DATE_ADD(created_at, INTERVAL banner_duration DAY) >= ?', [$now])
            ->get();
    
        return response()->json($banners);
    }

    /**
     * Get a single banner by ID.
     */
    public function getBannerById($id) {
        $banner = Banner::find($id);
        if (!$banner) {
            return response()->json(['message' => 'Banner not found'], 404);
        }
        return response()->json($banner);
    }

    /**
     * Create a new banner.
     */
    public function createBanner(Request $request) {
        $request->validate([
            'banner_duration' => 'required|integer',
            'expire_date' => 'required|date',
            'banner_bg_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'title' => 'required|string|max:255',
            'sub_title' => 'required|string|max:255',
            'btn_text' => 'required|string|max:255',
            'btn_link' => 'required|url|max:255',
        ]);
    
        // Upload image
        if ($request->hasFile('banner_bg_image')) {
            $filename = time() . '_' . $request->file('banner_bg_image')->getClientOriginalName();
            $request->file('banner_bg_image')->move(public_path('uploads/banner'), $filename);
            $imagePath = 'uploads/banner/' . $filename;
        } else {
            return response()->json(['message' => 'Image upload failed'], 500);
        }
    
        $banner = Banner::create([
            'banner_duration' => $request->banner_duration,
            'expire_date' => $request->expire_date,
            'banner_bg_image' => $imagePath,
            'title' => $request->title,
            'sub_title' => $request->sub_title,
            'btn_text' => $request->btn_text,
            'btn_link' => $request->btn_link,
        ]);
    
        return response()->json(['message' => 'Banner created successfully', 'banner' => $banner], 201);
    }
    

    /**
     * Update a banner.
     */
    public function updateBanner(Request $request, $id) {
        $banner = Banner::find($id);
        if (!$banner) {
            return response()->json(['message' => 'Banner not found'], 404);
        }
    
        $request->validate([
            'banner_duration' => 'integer',
            'expire_date' => 'date',
            'banner_bg_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'title' => 'string|max:255',
            'sub_title' => 'string|max:255',
            'btn_text' => 'string|max:255',
            'btn_link' => 'url|max:255',
        ]);
    
        if ($request->hasFile('banner_bg_image')) {
            // Delete old image if exists
            if ($banner->banner_bg_image && file_exists(public_path($banner->banner_bg_image))) {
                unlink(public_path($banner->banner_bg_image));
            }
    
            $filename = time() . '_' . $request->file('banner_bg_image')->getClientOriginalName();
            $request->file('banner_bg_image')->move(public_path('uploads/banner'), $filename);
            $imagePath = 'uploads/banner/' . $filename;
        } else {
            $imagePath = $banner->banner_bg_image;
        }
    
        $banner->update([
            'banner_duration' => $request->banner_duration ?? $banner->banner_duration,
            'expire_date' => $request->expire_date ?? $banner->expire_date,
            'banner_bg_image' => $imagePath,
            'title' => $request->title ?? $banner->title,
            'sub_title' => $request->sub_title ?? $banner->sub_title,
            'btn_text' => $request->btn_text ?? $banner->btn_text,
            'btn_link' => $request->btn_link ?? $banner->btn_link,
        ]);
    
        return response()->json(['message' => 'Banner updated successfully', 'banner' => $banner]);
    }
    

    /**
     * Delete a banner.
     */
    public function deleteBanner($id) {
        $banner = Banner::find($id);
        if (!$banner) {
            return response()->json(['message' => 'Banner not found'], 404);
        }
    
        // Delete image
        if ($banner->banner_bg_image) {
            $imagePath = public_path($banner->banner_bg_image);
            if (file_exists($imagePath)) {
                unlink($imagePath); // Use native PHP method to delete public image
            }
        }
    
        $banner->delete();
    
        return response()->json(['message' => 'Banner deleted successfully']);
    }
    
}
