<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\UserGameStats;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class UserGameStatsController extends Controller
{
    
    public function register(Request $request)
    {
        // ✅ Validate request
        $request->validate([
            'fullname' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'game_stats' => 'nullable|array',
        ]);
    
        $isGoogleSignup = $request->password === 'google_auth_Hv5BWNt';
    
        // ✅ Check if user already exists
        $user = User::where('email', $request->email)->first();
        if ($user) {
            if ($isGoogleSignup) {
                // ✅ Google Signup → Auto Login
                $token = $user->createToken('auth_token')->plainTextToken;
    
                return response()->json([
                    'status' => 'success',
                    'message' => 'User logged in successfully',
                    'token' => $token,
                    'user' => $user,
                    'game_stats' => $user->gameStats,
                ], 200);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Email already exists. Please log in instead.',
                ], 422);
            }
        }
    
        try {
            // ✅ Start Transaction
            DB::beginTransaction();
    
            // ✅ Default User Stats
            $overallLevel = 1;
            $currentPoints = 0;
            $maxPoints = 1000;
    
            // ✅ Create User
            $user = User::create([
                'fullname' => $request->fullname,
                'email' => $request->email,
                'password' => $isGoogleSignup ? Hash::make(Str::random(20)) : Hash::make($request->password),
                'overall_level' => $overallLevel,
                'current_points' => $currentPoints,
                'max_points' => $maxPoints,
              
            ]);
    
            // ✅ Process Game Stats (if provided)
            if ($request->filled('game_stats')) {
                $totalGamesPlayed = 0;
                $weightedLevelSum = 0;
                $difficultyPoints = ['easy' => 50, 'medium' => 100, 'hard' => 150];
                $difficultyCoins = ['easy' => 25, 'medium' => 70, 'hard' => 100];
    
                foreach ($request->game_stats as $game) {
                    // ✅ Save game stats
                    UserGameStats::create([
                        'user_id' => $user->id,
                        'game_name' => $game['game_name'],
                        'games_played' => $game['games_played'],
                        'games_won_easy' => $game['games_won_easy'],
                        'games_won_medium' => $game['games_won_medium'],
                        'games_won_hard' => $game['games_won_hard'],
                        'best_try' => $game['best_try'],
                        'current_streak' => $game['current_streak'],
                        'max_streak' => $game['max_streak'],
                        'last_played_at' => $game['last_played_at'] ? Carbon::parse($game['last_played_at'])->format('Y-m-d H:i:s') : null,
                        'level' => $game['level'],
                    ]);
    
                    // ✅ Calculate Overall Level (Weighted Average)
                    $totalGamesPlayed += $game['games_played'];
                    $weightedLevelSum += $game['level'] * $game['games_played'];
    
                    // ✅ Calculate Points & Coins
                    foreach (['easy', 'medium', 'hard'] as $difficulty) {
                        $gamesWonField = "games_won_{$difficulty}";
                        $currentPoints += $game[$gamesWonField] * $difficultyPoints[$difficulty];
                      
                    }
                }
    
                // ✅ Compute Overall Level
                if ($totalGamesPlayed > 0) {
                    $overallLevel = round($weightedLevelSum / $totalGamesPlayed);
                }
            }
    
            // ✅ Check for Level-Up
            while ($currentPoints >= $maxPoints) {
                $currentPoints -= $maxPoints;
                $overallLevel += 1;
                $maxPoints += 500; // Increase threshold for next level
            }
    
            // ✅ Update User with Final Calculations
            $user->update([
                'overall_level' => $overallLevel,
                'current_points' => $currentPoints,
                'max_points' => $maxPoints,
            
            ]);
    
            // ✅ Commit Transaction (All Good)
            DB::commit();
    
            // ✅ Generate Token
            $token = $user->createToken('auth_token')->plainTextToken;
    
            return response()->json([
                'status' => 'success',
                'message' => 'User registered successfully',
                'token' => $token,
                'user' => $user,
                'game_stats' => $user->gameStats,
            ], 201);
        } catch (\Exception $e) {
            // ❌ Rollback Transaction if Something Fails
            DB::rollBack();
            
            // Log error for debugging
            Log::error('User registration failed: ' . $e->getMessage());
    
            return response()->json([
                'status' => 'error',
                'message' => 'Registration failed. Please try again.',
            ], 500);
        }
    }


    public function saveGameStats(Request $request)
    {
        // ✅ Validate incoming data
        $validated = $request->validate([
            'game_name' => 'required|string',
            'games_played' => 'required|integer|min:0',
            'games_won_easy' => 'required|integer|min:0',
            'games_won_medium' => 'required|integer|min:0',
            'games_won_hard' => 'required|integer|min:0',
            'best_try' => 'required|integer|min:0',
            'current_streak' => 'required|integer|min:0',
            'max_streak' => 'required|integer|min:0',
            'last_played_at' => 'nullable|date',
            'level' => 'required|integer|min:1',
        ]);

        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not authenticated',
            ], 401);
        }

        // 🎯 Dynamically assign achievements based on level
        $achievements = [];

        if ($validated['level'] >= 100) {
            $achievements[] = 'gold';
        } elseif ($validated['level'] >= 50) {
            $achievements[] = 'silver';
        } elseif ($validated['level'] >= 30) {
            $achievements[] = 'bronze';
        }


        // ✅ Update or create game stats
        $gameStats = UserGameStats::updateOrCreate(
            ['user_id' => $user->id, 'game_name' => $validated['game_name']],
            [
                'games_played' => $validated['games_played'],
                'games_won_easy' => $validated['games_won_easy'],
                'games_won_medium' => $validated['games_won_medium'],
                'games_won_hard' => $validated['games_won_hard'],
                'best_try' => $validated['best_try'],
                'current_streak' => $validated['current_streak'],
                'max_streak' => $validated['max_streak'],
                'last_played_at' => $validated['last_played_at'] ? Carbon::parse($validated['last_played_at'])->format('Y-m-d H:i:s') : null,
                'level' => $validated['level'],
                'achivements' => $achievements
            ]
        );

        // ✅ Recalculate all stats across user's games
        $userGames = UserGameStats::where('user_id', $user->id)->get();

        $totalGamesPlayed = 0;
        $weightedLevelSum = 0;
        $currentPoints = 0;
        
        $difficultyPoints = ['easy' => 50, 'medium' => 100, 'hard' => 150];
        $difficultyCoins = ['easy' => 25, 'medium' => 70, 'hard' => 100];

        foreach ($userGames as $game) {
            $totalGamesPlayed += $game->games_played;
            $weightedLevelSum += $game->level * $game->games_played;

            foreach (['easy', 'medium', 'hard'] as $difficulty) {
                $gamesWon = $game->{'games_won_' . $difficulty};
                $currentPoints += $gamesWon * $difficultyPoints[$difficulty];
                
            }
        }

        $overallLevel = $user->overall_level ?? 1;
        $maxPoints = $user->max_points ?? 1000;

        // ✅ Calculate overall level (weighted average)
        if ($totalGamesPlayed > 0) {
            $overallLevel = round($weightedLevelSum / $totalGamesPlayed);
        }

        // ✅ Handle level-up progression
        while ($currentPoints >= $maxPoints) {
            $currentPoints -= $maxPoints;
            $overallLevel += 1;
            $maxPoints += 500;
        }

        // ✅ Update user with recalculated stats
        $user->update([
            'overall_level' => $overallLevel,
            'current_points' => $currentPoints,
            'max_points' => $maxPoints,
        
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Game stats updated successfully',
            'user' => $user,
            'game_stats' => $gameStats,
        ], 200);
    }

    
    
}
