<?php

namespace App\Http\Controllers\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\Settings\DepartmentDataTable;

class DepartmentController extends Controller
{
    public function department_list(DepartmentDataTable $dataTable)
    {
        return $dataTable->render('admin.settings.department');
    }
    public function add_department_form(Request $request)
{
    // Validate the incoming request
    $request->validate([
        'department' => 'required|string|max:255',
    ]);

    try {
        $token = session('bearer_token');
        $org_id = session('org_id');
      
        // Check if token or org_id are available
        if (!$token || !$org_id) {
            return response()->json([
                'success' => false,
                'message' => 'Missing authentication token or organization ID.'
            ], 400);
        }

        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->post("{$apiBaseUrl}/departments/store", [
            'department' => $request->department,
            'org_id' => $org_id
        ]);

        // Check if the request was successful
        if ($response->successful()) {
            return response()->json([
                'success' => true,
                'message' => 'Department added successfully'
            ]);
        } else {
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Failed to add Department.'
            ], $response->status());
        }
    } catch (\Exception $e) {
        // Catch any exceptions and return a formatted response
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong: ' . $e->getMessage()
        ], 500);
    }
}

    public function update_department_form(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send a PUT request to update the department
            $response = Http::withToken($token)->put("{$apiBaseUrl}/departments/update/{$id}", [
                'department' => $request->department,
            ]);
    
            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Department updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update department.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    
    
    public function delete_department($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/departments/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Department deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Department'
            ], $response->status());
        }
    }

    public function get_department($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/departments/get/{$id}");
    
          
    
            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['department']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Department not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    
    
    
}
