<?php

namespace App\Http\Controllers\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\Settings\DocumentDataTable;

class DocumentController extends Controller
{
    private function fetch_employees(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $employees = $response->json()['employees'] ?? [];

            // Extract only the employee ID and the associated user name
            $employees = array_map(function ($employee) {
                return [
                    'id' => $employee['id'], // Employee ID
                    'name' => $employee['user_profile']['user']['name'] ?? null, // Associated User's Name
                ];
            }, $employees);

            return $employees;
        }

        return [];
    }
    public function document_list(DocumentDataTable $dataTable)
    {
        $employees = $this->fetch_employees();
        return $dataTable->render('admin.settings.document', compact('employees'));
    }
    public function add_document_form(Request $request)
    {
        // Validate the input data, including the document file
        $request->validate([
            'name' => 'required',
            'document' => 'required|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
            'employee_id' => 'required', // Validate employee_id
            'status' => 'required|in:active,inactive', // Validate status
            'expiry_date' => 'required|after:now', // Validate remind_date if provided
            'remind_before' => 'required', // Validate remind_date if provided
        ], [
            'name.required' => 'The name field is required.',
            'document.required' => 'The document file is required.',
            'document.mimes' => 'The document must be a file of type: jpeg, png, jpg, pdf, doc, docx.',
            'document.max' => 'The document must not be greater than 2MB.',
            'employee_id.exists' => 'Selected employee does not exist.',
            'status.required' => 'Please select a status.',
            'status.in' => 'Invalid status selected.',
            'expiry_date.required' => 'The expiry date is required.',
            'expiry_date.after' => 'The expiry date must be a future date.',
        ]);

        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            // Prepare the request by attaching the file and sending other data as form fields
            $response = Http::withToken($token)
                ->attach('document', file_get_contents($request->file('document')->getRealPath()), $request->file('document')->getClientOriginalName())
                ->post("{$apiBaseUrl}/documents/store", [
                    'name' => $request->name,
                    'expiry_date' => $request->expiry_date,
                    'remind_before' => $request->remind_before,
                    'employee_id' => $request->employee_id, // Include employee_id
                    'status' => $request->status, // Include status
                    'org_id' => $org_id
                ]);
            // dd($response->json());

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Document added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Document.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function update_document_form(Request $request, $id)
    {
        // Validate the input data, including the document file
        $request->validate([
            'name' => 'required',
            'document' => 'required|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
            // Validate employee_id
            'status' => 'required|in:active,inactive', // Validate status
            'expiry_date' => 'required|after:now', // Validate remind_date if provided
            'remind_before' => 'required', // Validate remind_date if provided
        ], [
            'name.required' => 'The name field is required.',
            'document.required' => 'The document file is required.',
            'document.mimes' => 'The document must be a file of type: jpeg, png, jpg, pdf, doc, docx.',
            'document.max' => 'The document must not be greater than 2MB.',
            'status.required' => 'Please select a status.',
            'status.in' => 'Invalid status selected.',
            'expiry_date.required' => 'The expiry date is required.',
            'expiry_date.after' => 'The expiry date must be a future date.',
        ]);
    
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Prepare the request by attaching the file if it exists
            $requestData = [
                'name' => $request->name,
                'expiry_date' => $request->expiry_date,
                'remind_before' => $request->remind_before,
                'employee_id' => $request->employee_id, // Include employee_id if provided
                'status' => $request->status, // Include status
            ];
    
            $http = Http::withToken($token)->timeout(60);
    
            if ($request->hasFile('document')) {
                $http = $http->attach('document', file_get_contents($request->file('document')->getRealPath()), $request->file('document')->getClientOriginalName());
            }
    
            $response = $http->post("{$apiBaseUrl}/documents/update/{$id}", $requestData);
    
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Document updated successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update document.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    


    public function delete_document($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/documents/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Document deleted successfully']);
        } else {
            // Extract error message from the API response, defaulting to a generic message
            $error = $response->json();
            $errorMessage = $error['message'] ?? 'Unable to delete Document';

            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], $response->status());
        }
    }
    public function get_document($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/documents/get/{$id}");



            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['document']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Document not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
