<?php

namespace App\Http\Controllers\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\Settings\LeaveTypeDataTable;

class LeaveTypeController extends Controller
{
    public function leave_type_list(LeaveTypeDataTable $dataTable)
    {
        return $dataTable->render('admin.settings.leave_type');
    }
    public function add_leave_type_form(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            
            // API request to store leave type
            $response = Http::withToken($token)->post("{$apiBaseUrl}/leave/types/store", [
                'name' => $request->name,
                'status' => 'active',
                'org_id' => $org_id
            ]);
    
            // Check if the API response is successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Leave Type added successfully'
                ]);
            } else {
                // Get the error message from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Leave Type.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle any errors that occur in the try block
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    
    public function update_leave_type_form(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send a PUT request to update the leave type
            $response = Http::withToken($token)->put("{$apiBaseUrl}/leave/types/update/{$id}", [
                'name' => $request->name,
                'status' => 'active',
            ]);
    
            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Leave Type updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                // Check if the API response contains validation errors
                if (isset($error['errors'])) {
                    $errors = $error['errors'];
                    return response()->json([
                        'success' => false,
                        'message' => $errors
                    ], $response->status());
                }
    
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Leave Type.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    
    
    public function delete_leave_type($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/leave/types/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Leave Type deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Leave Type'
            ], $response->status());
        }
    }

    public function get_leave_type($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/leave/types/get/{$id}");
    
          
    
            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['leave_type']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Leave Type not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    public function update_leave_type_status(Request $request)
    {
        try {
            // Validate incoming request
            $request->validate([
                'id' => 'required|integer', // Ensure 'id' is present and is an integer
            ]);
    
            // Get the bearer token from the session and the base API URL from environment config
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            
            // Prepare the payload with table and ID
            $payload = [
                'table' => 'leave_types',  // The table name is hardcoded as 'organizations'
                'id' => $request->id  // ID is passed from the request
            ];
            
            // Send a POST request to the external API to update the status
            $response = Http::withToken($token)->post("{$apiBaseUrl}/update/status", $payload);
    
            // Check if the response was successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Status updated successfully'
                ]);
            } else {
                // If the request fails, return the error message from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update status'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Catch any exceptions and return a 500 response with the error message
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
}
