<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\DataTables\LeaveDataTable;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;

class LeaveController extends Controller
{
    private function fetch_leave_type(): array
    {
        $token = session('bearer_token');
        $org_id = session('org_id');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/leave/types");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $leave_types = $response->json()['leave_types'] ?? [];

            // Extract only the employee ID and the associated user name
            $leave_types = array_map(function ($leave_types) {
                return [
                    'id' => $leave_types['id'], // Employee ID
                    'name' => $leave_types['name'], // Associated User's Name
                ];
            }, $leave_types);

            return $leave_types;
        }

        return [];
    }
    
    public function leave_list()
    {

        $leave_types = $this->fetch_leave_type();
        return view('admin.employee.leave', compact('leave_types'));
    }

    private function fetch_employees(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $employees = $response->json()['employees'] ?? [];

            // Extract only the employee ID and the associated user name
            $employees = array_map(function ($employee) {
                return [
                    'id' => $employee['user_profile']['user']['id'] ?? null, // Employee ID
                    'name' => $employee['user_profile']['user']['name'] ?? null, // Associated User's Name
                ];
            }, $employees);

            return $employees;
        }

        return [];
    }

    public function admin_leave_list()
    {
        $leave_types = $this->fetch_leave_type();
        $employees = $this->fetch_employees();
        return view('admin.leave.leave', compact('employees', 'leave_types'));
    }

    public function add_leave_form(Request $request)
    {
        // Define validation rules
        $validatedData = $request->validate([
            'leave_type_id' => 'required|integer',
            'reason' => 'required|string|max:255',
            'start_date' => 'required',
            'end_date' => 'required|after_or_equal:start_date',
        ]);

        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $user_id = session('id'); // Get user_id from session

            // API request to store leave
            $response = Http::withToken($token)->post("{$apiBaseUrl}/leaves/store", [
                'leave_type_id' => $validatedData['leave_type_id'],
                'reason' => $validatedData['reason'],
                'start_date' => $validatedData['start_date'],
                'end_date' => $validatedData['end_date'],
                'org_id' => $org_id,
                'user_id' => $user_id, // Pass user_id from session
            ]);

            // Check if the API response is successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Leave added successfully'
                ]);
            } else {
                // Get the error message from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Leave.'
                ], $response->status());
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(), // Return detailed validation errors
            ], 422);
        } catch (\Exception $e) {
            // Handle any other errors that occur
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }


    public function add_admin_leave_form(Request $request)
    {
        // Define validation rules
        $validatedData = $request->validate([
            'leave_type_id' => 'required|integer',
            'reason' => 'required|string|max:255',
            'start_date' => 'required',
            'end_date' => 'required|after_or_equal:start_date',
            'user_id' => 'required'
        ]);

        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');

            // API request to store leave
            $response = Http::withToken($token)->post("{$apiBaseUrl}/leaves/store", [
                'leave_type_id' => $validatedData['leave_type_id'],
                'reason' => $validatedData['reason'],
                'start_date' => $validatedData['start_date'],
                'end_date' => $validatedData['end_date'],
                'status' => 'approve',
                'leave_status' => 'paid',
                'org_id' => $org_id,
                'user_id' => $validatedData['user_id'], // Pass user_id from session
            ]);

            // Check if the API response is successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Leave added successfully'
                ]);
            } else {
                // Get the error message from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Leave.'
                ], $response->status());
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(), // Return detailed validation errors
            ], 422);
        } catch (\Exception $e) {
            // Handle any other errors that occur
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }



    public function update_leave_form(Request $request, $id)
    {
        // Add validation
        $validatedData = $request->validate([
            'leave_type_id' => 'required',
            'reason' => 'required|string|max:255',
            'start_date' => 'required',
            'end_date' => 'required|after_or_equal:start_date',
        ]);

        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $user_id = session('id'); // Get user_id from session

            // Send a PUT request to update the leave
            $response = Http::withToken($token)->put("{$apiBaseUrl}/leaves/update/{$id}", [
                'leave_type_id' => $validatedData['leave_type_id'],
                'reason' => $validatedData['reason'],
                'start_date' => $validatedData['start_date'],
                'end_date' => $validatedData['end_date'],
                'user_id' => $user_id
            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Leave updated successfully']);
            } else {
                $error = $response->json();

                // Check if the API response contains validation errors
                if (isset($error['errors'])) {
                    return response()->json([
                        'success' => false,
                        'message' => array_values($error['errors'])[0][0] // First validation error
                    ], $response->status());
                }

                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update leave.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    public function leave_approve(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Send a PUT request to update the leave status
            $response = Http::withToken($token)->put("{$apiBaseUrl}/leaves/update/{$id}", [
                'status' => 'approve',
                'user_id' => $request->user_id,
                'leave_type_id' => $request->leave_type_id,
                'leave_status' => $request->leave_status, // Paid or Unpaid
            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Leave Approved Successfully'
                ]);
            } else {
                // Check for validation errors in the API response
                $error = $response->json();
                if (isset($error['errors'])) {
                    return response()->json([
                        'success' => false,
                        'errors' => $error['errors']
                    ], 422);
                }

                // Handle general errors
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to approve leave.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle any general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }

    public function leave_reject(Request $request, $id)
    {


        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');


            // Send a PUT request to update the leave
            $response = Http::withToken($token)->put("{$apiBaseUrl}/leaves/update/{$id}", [
                'status' => 'reject',
                'user_id' => $request->user_id,
                'leave_type_id' => $request->leave_type_id,
                'leave_status' => 'reject',

            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return redirect()->back()->with('success', 'Leave Rejected Successfully');
            } else {
                $error = $response->json();

                // Check if the API response contains validation errors
                if (isset($error['errors'])) {
                    return response()->json([
                        'success' => false,
                        'message' => array_values($error['errors'])[0][0] // First validation error
                    ], $response->status());
                }

                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to Reject leave.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }




    public function delete_leave($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/leaves/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Leave  deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Leave '
            ], $response->status());
        }
    }

    public function get_leave($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/leaves/get/{$id}");



            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['leave']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Leave  not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    public function get_total_leaves($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            
            // Send the request to the API with the bearer token
            $response = Http::withToken($token)->get("{$apiBaseUrl}/user/leaves/{$id}");
    
            if ($response->successful()) {
                // Retrieve the data from the response
                $leavesData = $response->json();
    
                // Return the specific leave data in a JSON format
                return response()->json([
                    'success' => true,
                    'message' => $leavesData['message'],
                    'total_allocated_leaves' => $leavesData['total_allocated_leaves'],
                    'used_leaves' => $leavesData['used_leaves'],
                    'remaining_leaves' => $leavesData['remaining_leaves']
                ]);
            } else {
                return response()->json(['status' => 404, 'message' => 'Leaves not found.'], 404);
            }
        } catch (\Exception $e) {
            // Return a JSON response for any exceptions that occur
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
