<?php

namespace App\Http\Controllers\Settings;

use SplTempFileObject;
use Illuminate\Http\Response;
use App\DataTables\ExpenseRequestDataTable;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use League\Csv\Writer;
class IncomeExpenseController extends Controller
{
    private function fetch_employees(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $employees = $response->json()['employees'] ?? [];

            // Extract only the employee ID and the associated user name
            $employees = array_map(function ($employee) {
                return [
                    'id' => $employee['id'] ?? null, // Employee ID
                    'name' => $employee['user_profile']['user']['name'] ?? null, // Associated User's Name
                ];
            }, $employees);

            return $employees;
        }

        return [];
    }
    public function expense_list()
    {
        $employees = $this->fetch_employees();
        return view('admin.settings.expense', compact('employees'));
    }
    public function add_expense_form(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $response = Http::withToken($token)->post("{$apiBaseUrl}/incomes/store", [
                'date' => $request->date,
                'description' => $request->description,
                'amount' => $request->amount,
                'status' => $request->income_expence_type == 'petty_cash' ? 'pending' : 'approve',  
                'type' => 'expense', // hardcoded as "expense"
                'employee_id' => $request?->employee_id ,
                'income_expence_type' => $request->income_expence_type,
                'org_id' => $org_id
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Expense added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add expense.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function update_expense_form(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Send a PUT request to update the expense
            $response = Http::withToken($token)->put("{$apiBaseUrl}/incomes/update/{$id}", [
                'date' => $request->date,
                'description' => $request->description,
                'amount' => $request->amount,
                'status' => 'approve',   // assuming status should come from the request
                'type' => 'expense', // hardcoded as "expense"
            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Expense updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Expense.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }


    public function delete_expense($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/incomes/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Expense deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Expense'
            ], $response->status());
        }
    }

    public function get_expense($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/incomes/get/{$id}");



            if ($response->successful()) {
                // Return the expense data from the API response
                return response()->json($response->json()['income']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Expense not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    public function income_list()
    {
        $employees = $this->fetch_employees();
        return view('admin.settings.income', compact('employees'));
    }

    public function add_income_form(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $response = Http::withToken($token)->post("{$apiBaseUrl}/incomes/store", [
                'date' => $request->date,
                'description' => $request->description,
                'amount' => $request->amount,
                'status' => 'approve', 
                'type' => 'income',
                'employee_id' => $request?->employee_id ,
                'income_expence_type' => $request->income_expence_type,
                'org_id' => $org_id
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Income added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Income.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function update_income_form(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Send a PUT request to update the expense
            $response = Http::withToken($token)->put("{$apiBaseUrl}/incomes/update/{$id}", [
                'date' => $request->date,
                'description' => $request->description,
                'amount' => $request->amount,
                'status' => 'approve',  // assuming status should come from the request
                'type' => 'income', // hardcoded as "expense"
            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Income updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Income.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    public function delete_income($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/incomes/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Income deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Income'
            ], $response->status());
        }
    }
    public function employee_expense_request(ExpenseRequestDataTable $dataTable)
    {
        return $dataTable->render('admin.expense_request.employee_expense_request');
    }
    public function admin_expense_request()
    {
        $employees = $this->fetch_employees();
        return view('admin.expense_request.admin_expense_request', compact('employees'));
    }
    public function expense_request_form(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $employee_id = session('employee_id');
            $response = Http::withToken($token)->post("{$apiBaseUrl}/incomes/store", [
                'employee_id' => $employee_id,
                'date' => $request->date,
                'description' => $request->description,
                'amount' => $request->amount,
                'status' => 'pending',  // assuming status should come from the request
                'type' => 'employee_expense', // hardcoded as "expense"
                'org_id' => $org_id
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Expense added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add expense.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    public function expense_approve(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Send a PUT request to update the leave status
            $response = Http::withToken($token)->put("{$apiBaseUrl}/incomes/update/{$id}", [
                'status' => 'approve',
            ]);
            if ($response->successful()) {
                return redirect()->back()->with('success','Expense Approved Successfully');
            } else {
                // Check for validation errors in the API response
                $error = $response->json();
                if (isset($error['errors'])) {
                    return response()->json([
                        'success' => false,
                        'errors' => $error['errors']
                    ], 422);
                }

                // Handle general errors
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to approve Expense.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle any general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }

    public function expense_reject(Request $request, $id)
    {


        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');


            // Send a PUT request to update the leave
            $response = Http::withToken($token)->put("{$apiBaseUrl}/incomes/update/{$id}", [
                'status' => 'reject',
            ]);
            if ($response->successful()) {
                return redirect()->back()->with('success', 'Expense Rejected Successfully');
            } else {
                $error = $response->json();

                // Check if the API response contains validation errors
                if (isset($error['errors'])) {
                    return response()->json([
                        'success' => false,
                        'message' => array_values($error['errors'])[0][0] // First validation error
                    ], $response->status());
                }

            return redirect()->back()->with('success','Expense Rejected Successfully');
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    public function export_incomes_csv(Request $request)
    {
        $validatedData = $request->validate([
            'month' => 'required|date_format:Y-m',
        ]);
    
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
    
        $response = Http::withToken($token)->get("{$apiBaseUrl}/incomes/search?type=income&month={$validatedData['month']}");
    
        // Check if the response is successful
        if ($response->successful()) {
            $incomes = $response->json()['incomes'];
    
            // Check if there are no incomes
            if (empty($incomes)) {
                return response()->json(['success' => false, 'message' => 'No incomes found for the specified criteria.'], Response::HTTP_NOT_FOUND);
            }
    
            // Create a new CSV writer instance
            $csv = Writer::createFromFileObject(new SplTempFileObject());
    
            // Insert the header
            $csv->insertOne(['Date', 'Amount', 'Description']);
    
            // Insert income data
            foreach ($incomes as $income) {
                $csv->insertOne([
                    $income['date'],
                    $income['amount'],
                    $income['description'],
                ]);
            }
    
            // Create a response for download
            return response((string) $csv)
                ->header('Content-Type', 'text/csv')
                ->header('Content-Disposition', 'attachment; filename="income.csv"');
        }
    
        // Handle error response
        return response()->json(['error' => 'Unable to fetch income data'], Response::HTTP_BAD_REQUEST);
    }
    public function export_expenses_csv(Request $request)
    {
        $validatedData = $request->validate([
            'month' => 'required|date_format:Y-m',
        ]);
    
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
    
        $response = Http::withToken($token)->get("{$apiBaseUrl}/incomes/search?type=expense&month={$validatedData['month']}");
    
        // Check if the response is successful
        if ($response->successful()) {
            $incomes = $response->json()['incomes'];
            $csv = Writer::createFromFileObject(new SplTempFileObject());
    
            // Insert the header
            $csv->insertOne(['Date', 'Amount', 'Description']);
    
            // Insert income data
            foreach ($incomes as $income) {
                $csv->insertOne([
                    $income['date'],
                    $income['amount'],
                    $income['description'],
                ]);
            }
    
            // Create a response for download
            return response((string) $csv)
                ->header('Content-Type', 'text/csv')
                ->header('Content-Disposition', 'attachment; filename="income.csv"');
        }
    
        // Handle error response
        return response()->json(['error' => 'Unable to fetch income data'], Response::HTTP_BAD_REQUEST);
    }
}
