<?php

namespace App\Http\Controllers\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;

class PettyCashController extends Controller
{
    private function fetch_employees(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        
        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $employees = $response->json()['employees'] ?? [];

            // Extract only the employee ID and the associated user name
            $employees = array_map(function ($employee) {
                return [
                    'id' => $employee['id'], // Employee ID
                    'name' => $employee['user_profile']['user']['name'] ?? null, // Associated User's Name
                ];
            }, $employees);

            return $employees;
        }

        return [];
    }

    public function index()
    {
        try {
            return view('admin.employee.petty_cash');
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $employee_id = session('employee_id');

            // Create new petty cash entry
            $response = Http::withToken($token)->post("{$apiBaseUrl}/petty/cashes", [
                'amount' => $request->amount,
                'date' => $request->date,
                'description' => $request->description,
                'employee_id' => $employee_id,
            ]);
           
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Petty cash added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add petty cash.'
                ], 200);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Get a single petty cash entry
            $response = Http::withToken($token)->get("{$apiBaseUrl}/petty/cashes/{$id}");

            if ($response->successful()) {
                $pettyCashData = $response->json()['petty_cash'];
                return response()->json($pettyCashData);
            } else {
                return response()->json(['status' => 404, 'message' => 'Petty cash not found.'], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Update petty cash entry
            $employee_id = session('employee_id');
            $response = Http::withToken($token)->put("{$apiBaseUrl}/petty/cashes/{$id}", [
                'employee_id' =>  $employee_id,
                'amount' => $request->amount,
                'date' => $request->date,
                'description' => $request->description,
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Petty cash updated successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update petty cash.'
                ], 200);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Delete petty cash entry
            $response = Http::withToken($token)->delete("{$apiBaseUrl}/petty/cashes/{$id}");

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Petty cash deleted successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to delete petty cash.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
}

