<?php

namespace App\Http\Controllers\Api\Auth;

use App\Models\User;
use App\Models\Settings\UserProfile;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Settings\Employee;
use App\Models\Settings\Organization;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\ValidationException;
use App\Models\Settings\Document;

use function PHPUnit\Framework\isNull;

class AuthController extends Controller
{

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);
        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json(['message' => 'Invalid login details'], 401);
        }
        // Fetch user with organization, userProfile, and employee relation
        $user = User::with(['organization', 'userProfile.employees.designation'])->where('email', $request->email)->firstOrFail();
        $employee = $user->userProfile ? $user->userProfile->employees->first() : null;
        $employeeId = $employee ? $employee->id : null;
        $designation = $employee ? $employee->designation : null;
        // Check if the user's status is inactive
        if ($user->status === 'inactive') {
            return response()->json(['message' => 'Account is inactive'], 401);
        }
        // Generate token for active users
        $token = $user->createToken('auth_token')->plainTextToken;
        $response = [
            'message' => 'Login Successfully',
            'access_token' => $token,
            'token_type' => 'Bearer',
            'role' => $user->role,
            'permissions' => $this->getPermissions($user),
            'logo' => isset($user->organization->image) ? asset($user->organization->image) : null,
            'currency' => $user->organization->currency ?? '',
            'org_id' => $user->org_id ?? '',
            'user_name' => $user->name ?? '',
            'id' => $user->id,
            'employee_id' => $employeeId,
            'designation' => $designation ? $designation->designation : null, // Include designation name
        ];
        return response()->json($response, 200);
    }

    protected function getPermissions($user)
    {
        // If user has specific permissions, return those
        if (isset($user->permissions)) {
            return $user->permissions;
        }
        if ($user->role === 'employee') {
            return [];
        }
        if ($user->role != 'normal' && isset($user->organization->permissions)) {
            return $user->organization->permissions;
        }
       

        return [];
    }


    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'message' => 'Logged out successfully'
        ]);
    }
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'role' => 'admin',
            'password' => Hash::make($request->password),
        ]);

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'message' => 'User registered successfully',
            'access_token' => $token,
            'token_type' => 'Bearer',
        ], 201);
    }

    

    public function updateStatus(Request $request)
{
    try {
        // Validate the incoming request
        $validated = $request->validate([
            'table' => 'required|string',
            'id' => 'required|integer',
        ]);

        $table = $validated['table'];
        $id = $validated['id'];

        // Check if the table exists
        if (!Schema::hasTable($table)) {
            return response()->json([
                'success' => false,
                'message' => 'Table does not exist.'
            ], 400);
        }

        // Retrieve the current record to get the current status
        $record = DB::table($table)->where('id', $id)->first();

        // Check if the record exists
        if (!$record) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found.',
            ], 404);
        }

        // Invert the status: if "active", make it "inactive" and vice versa
        $currentStatus = $record->status;
        $newStatus = ($currentStatus === 'active') ? 'inactive' : 'active';

        // Attempt to update the record with the new status
        $updated = DB::table($table)
            ->where('id', $id)
            ->update(['status' => $newStatus]);

        // Check if the update was successful
        if ($updated) {
            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully.',
                'new_status' => $newStatus,
            ], 200);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status.',
            ], 400);
        }
    } catch (ValidationException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Record not found.',
        ], 404);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to update status.',
            'details' => $e->getMessage()
        ], 500);
    }
}

public function dashboard()
    {
        $org_id = Auth::user()->org_id;
        //dd($org_id);
        if(Auth::user()->role == 'admin'){
            $totalOrganizations = Organization::where('id', '!=' ,Auth::user()->org_id)->count();
            $totalOrganizationsLastMonth = Organization::where('id', '!=' ,Auth::user()->org_id)->where('created_at', '>=', Carbon::now()->subMonth())->count();
            $totalUsers = User::where('id', '!=' ,Auth::user()->org_id)->whereIn('role', ['normal', 'organization'])->count();
            $totalUsersLastMonth = User::where('id', '!=' ,Auth::user()->org_id)->whereIn('role', ['normal', 'organization'])
            ->where('created_at', '>=', Carbon::now()->subMonth())
            ->count();
    
            // Return the response as JSON
            return response()->json([
                'message'=> 'successfully fetched',
                'totalOrganizations' => $totalOrganizations,
                'totalOrganizationsLastMonth' => $totalOrganizationsLastMonth,
                'totalUsers' => $totalUsers,
                'totalUsersLastMonth' => $totalUsersLastMonth,
            ]);
        }else{
            $totalProfiles = UserProfile::where('org_id', $org_id)->count();
            $salaries = Employee::where('org_id', $org_id)->sum('salary');
            $today = Carbon::now();

            $documentsWithReminders = Document::where('org_id', $org_id)
                ->whereRaw('DATE_SUB(expiry_date, INTERVAL remind_before DAY) <= ?', [$today->format('Y-m-d')])  // Reminder is due
                ->where('expiry_date', '>=', $today->format('Y-m-d'))  // Expiry is still in the future
                ->count();

            $lastWeekProfiles = UserProfile::where('org_id', $org_id)->where('created_at', '>=', Carbon::now()->subWeek())->count();


            $nextBirthday = UserProfile::with('user')
                ->where('org_id', $org_id)
                ->select('phone', 'address', 'date_of_birth','user_id','image') 
                ->where(function($query) use ($today) {
                    $query->whereRaw("DATE_FORMAT(date_of_birth, '%m-%d') >= ?", [$today->format('m-d')]);
                })
                ->orderByRaw("DATE_FORMAT(date_of_birth, '%m-%d') ASC")
                ->first();

            if ($nextBirthday) {
                $birthday = [
                    'message' => 'Upcoming birthday found',
                    'employee' => [
                        'name' => $nextBirthday->user->name, 
                        'phone' => $nextBirthday->phone,
                        'image' => asset($nextBirthday->image),
                        'address' => $nextBirthday->address,
                        'date_of_birth' => $nextBirthday->date_of_birth,
                    ]
                ];
            } else {
                $birthday = [
                    'message' => 'No upcoming birthday found',
                ];
            }

            // Return the response as JSON
            return response()->json([
                'message'=> 'successfully fetched',
                'total_profiles' => $totalProfiles,
                'last_week_profiles' => $lastWeekProfiles,
                'salaries' => $salaries,
                'documentsWithReminders' => $documentsWithReminders,
                'birthday' => $birthday
            ]);
        }
        
    }

    public function reminders()
    {
        $org_id = Auth::user()->org_id;
        $today = Carbon::now();
    
        // Fetch the documents with reminders and necessary relationships
        $documentsWithReminders = Document::with(['employee.userProfile.user'])
            ->where('org_id', $org_id)
            ->whereRaw('DATE_SUB(expiry_date, INTERVAL remind_before DAY) <= ?', [$today->format('Y-m-d')])
            ->where('expiry_date', '>=', $today->format('Y-m-d'))
            ->get()
            ->map(function ($document) {
                return [
                    'id' => $document->id,
                    'name' => $document->name,
                    'document_url' => asset($document->document), // Full URL for download
                    'expiry_date' => $document->expiry_date,
                    'employee_name' => $document->employee->userProfile->user->name,
                ];
            });
    
        return response()->json([
            'message' => 'successfully fetched',
            'documentsWithReminders' => $documentsWithReminders,
        ]);
    }
    

}
