<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Deducation;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class DeducationController extends Controller
{
    public function get_deductions_index()
    {
        try {
            $org_id = Auth::user()?->org_id;
            $deductions = Deducation::where('org_id', $org_id)
                ->with('employee.userProfile.user') // Assuming Deducation model has a relationship with Employee
                ->get();
    
            return DataTables::of($deductions)
            ->addColumn('count', function($row) {
                static $count = 0;
                return ++$count;
            })
            ->editColumn('employee_name', function($row) {
                return $row->employee->userProfile->user->name ?? 'N/A';// Fetch employee name from related employee model
            })
            ->editColumn('amount', function($row) {
                return number_format($row['amount'], 2); // Format amount
            })
            ->editColumn('date', function($row) {
                return \Carbon\Carbon::parse($row['date'])->format('d-M-Y'); // Format date to d-M-Y format
            })
            ->editColumn('reason', function($row) {
                return $row['reason'] ?? ''; // Limit reason string to 50 characters
            })
            ->editColumn('status', function($row) {
                return $row['status'] ?? ''; 
            })
            ->rawColumns(['status'])
            ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function get_deductions()
    {
        try {
            $org_id = Auth::user()->org_id;
            $deductions = Deducation::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->get()
            ->map(function ($increment) {
                return [
                    'id' => $increment->id,
                    'employee_id' => $increment->employee_id,
                    'amount' => $increment->amount,
                    'date' => $increment->date,
                    'reason' => $increment->reason,
                    'status' => $increment->status,
                    'employee_name' => $increment->employee->userProfile->user->name ?? 'N/A',
                ];
            });
            return response()->json([
                "message" => "Deductions fetched successfully",
                "deductions" => $deductions
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch deductions.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_deduction(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
                'org_id' => 'required|exists:organizations,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $deduction = new Deducation();
            $deduction->employee_id = $request->employee_id;
            $deduction->amount = $request->amount;
            $deduction->date = $request->date;
            $deduction->reason = $request->reason;
            $deduction->org_id = $request->org_id;
            $deduction->status = $request->status;
            $deduction->save();

            return response()->json([
                'success' => true,
                'message' => 'Deducation created successfully',
                'deduction' => $deduction
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create deducation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_deduction($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $deduction = Deducation::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->findOrFail($id);
           
            return response()->json([
                'message' => 'Deducation retrieved successfully',
                'deduction' => $deduction
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Deducation not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve deducation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_deduction(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $deduction = Deducation::findOrFail($id);

            $deduction->employee_id = $request->employee_id;
            $deduction->amount = $request->amount;
            $deduction->date = $request->date;
            $deduction->reason = $request->reason;
            $deduction->status = $request->status;
            $deduction->save();

            return response()->json([
                'success' => true,
                'message' => 'Deducation updated successfully',
                'deduction' => $deduction
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Deducation not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update deducation',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_deduction($id)
    {
        try {
            $deduction = Deducation::findOrFail($id);

            $deduction->delete();

            return response()->json(['message' => 'Deducation deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Deducation not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete deducation',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
