<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Incomes;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class IncomeController extends Controller
{
    public function search_incomes_index(Request $request)
    {
        try {
            $org_id = Auth::user()->org_id;
    
            $query = Incomes::with('employee.UserProfile.user')->where('org_id', $org_id);
    
            // Filter by type if provided
            if ($request->has('income_expence_type') && $request->input('income_expence_type') != 'all') {
                $query->where('income_expence_type', $request->input('income_expence_type'));
            }

            if ($request->has('type') && $request->input('type') != 'all') {
                $query->where('type', $request->input('type'));
            }
    
            // Filter by employee ID if provided
            if ($request->has('employee_id')) {
                $query->where('employee_id', $request->input('employee_id'));
            }
    
            // Filter by month if provided
            if ($request->has('month')) {
                $month = $request->input('month');
                $query->whereYear('created_at', date('Y', strtotime($month)))
                      ->whereMonth('created_at', date('m', strtotime($month)));
            }
    
            // Fetch the incomes
            $incomes = $query->get();

            // Return the results as DataTables, with a success message
            return DataTables::of($incomes)
                ->addColumn('count', function($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->addColumn('employee_name', function ($income) {
                    return $income->employee->userProfile->user->name ?? 'N/A';
                })
                ->editColumn('date', function ($income) {
                    return \Carbon\Carbon::parse($income->date)->format('d-M-Y');
                })
                ->editColumn('description', function ($income) {
                    $userName = $income->employee->userProfile->user->name ?? null;
                    
                    if ($userName) {
                        return $income->description . ' (Employee - ' . $userName . ')';
                    }
                
                    return $income->description;
                })
                
                ->with('message', 'Incomes from datatable fetched successfully') // Add a custom response
                ->make(true);
    
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch incomes. ' . $e->getMessage(),
            ], 500);
        }
    }
    
    
    
    public function get_incomes()
    {
        try {
            $org_id = Auth::user()->org_id;
            $incomes = Incomes::with('employee.UserProfile.user')->where('org_id',$org_id)->get();

            return response()->json([
                "message" => "Incomes fetched successfully",
                "incomes" => $incomes
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch incomes.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function search_by_type(Request $request)
{
    try {
        $type = $request->input('type');
        $month = $request->input('month');
        $employeeId = $request->input('employee_id');

        if (!$type) {
            return response()->json([
                'success' => false,
                'message' => 'Please provide an income type to search.'
            ], 400);
        }

        $org_id = Auth::user()->org_id;

        // Build the query
        $query = Incomes::with('employee.UserProfile.user')
            ->where('org_id', $org_id)
            ->where('type', '=', $type);

        // Add condition if employee_id is provided
        if ($employeeId) {
            $query->where('employee_id', $employeeId);
        }

        // Filter by month if provided
        if ($month) {
            $query->whereYear('created_at', date('Y', strtotime($month)))
                  ->whereMonth('created_at', date('m', strtotime($month)));
        }

        $incomes = $query->get();

        // Return response based on the result
        if ($incomes->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No incomes found for the specified criteria.'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Incomes fetched successfully',
            'incomes' => $incomes
        ], 200);

    } catch (Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch incomes.',
            'details' => $e->getMessage()
        ], 500);
    }
}

    


    public function add_income(Request $request)
    {   

        try {
            $validator = Validator::make($request->all(), [
                'date' => 'required|date',
                'description' => 'required|string',
                'type' => 'required|string|max:50',
                'amount' => 'required|numeric',
                'status' => 'required|string|in:pending,approve,reject',
                'org_id' => 'required|exists:organizations,id',
                'employee_id' => 'nullable|exists:employee,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $income = new Incomes();
            $income->date = $request->date;
            $income->description = $request->description;
            $income->type = $request->type;
            $income->org_id = $request->org_id;
            $income->income_expence_type = $request?->income_expence_type;
            $income->amount = $request->amount;
            $income->status = $request->status;
            $income->employee_id = $request?->employee_id;
            $income->save();

            return response()->json([
                'success' => true,
                'message' => 'Income created successfully',
                'income' => $income
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create income.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_income($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $income = Incomes::with('employee.UserProfile.user')->where('org_id',$org_id)->findOrFail($id);
            return response()->json([
                'message' => 'Income retrieved successfully',
                'income' => $income
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Income not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve income.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_income(Request $request, $id)
{
    try {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'date' => 'nullable|string',
            'description' => 'nullable|string',
            'type' => 'nullable|string|max:50',
            'amount' => 'nullable|numeric',
            'status' => 'nullable|string|in:pending,approve,reject,complete',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        // Find the income record by ID, throw exception if not found
        $income = Incomes::findOrFail($id);

        // Use the fill method to update only provided fields
        $income->fill($request->only([
            'date',
            'description',
            'type',
            'amount',
            'status'
        ]));

        // Save the updated income record
        $income->save();

        return response()->json([
            'success' => true,
            'message' => 'Income updated successfully',
            'income' => $income
        ], 200);
    } catch (ValidationException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Income not found',
        ], 404);
    } catch (Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to update income',
            'details' => $e->getMessage()
        ], 500);
    }
}


    public function delete_income($id)
    {
        try {
            $income = Incomes::findOrFail($id);

            $income->delete();

            return response()->json(['message' => 'Income deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Income not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete income',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
