<?php

namespace App\Http\Controllers\Api;

use App\Models\Settings\Leave;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use DataTables;
class LeaveController extends Controller
{
 
    public function get_leaves_index()
{
    try {
        $org_id = Auth::user()->org_id;
        if (Auth::user()->role == 'employee') {
            $leaves = Leave::where('user_id', Auth::user()->id)
                            ->where('org_id', $org_id)
                            ->with(['user', 'leaveType'])
                            ->get();
        } else {
            $leaves = Leave::where('org_id', $org_id)
                            ->with(['user', 'leaveType'])
                            ->get();
        }

        // For DataTables, format the response
        return DataTables::of($leaves)
            ->addColumn('count', function($row) {
                static $count = 0;
                return ++$count;
            })
            ->addColumn('employee_name', function ($leave) {
                return $leave->user->name;
            })
            ->addColumn('leave_type', function ($leave) {
                return $leave->leaveType->name;
            })
            ->editColumn('start_date', function ($leave) {
                return $leave->start_date;
            })
            ->editColumn('end_date', function ($leave) {
                return $leave->end_date;
            })
            ->make(true);

    } catch (Exception $e) {
        return response()->json([
            'error' => 'Failed to fetch leaves.',
            'details' => $e->getMessage()
        ], 500);
    }
}

    public function get_leaves()
    {
        try {
            $org_id = Auth::user()->org_id;
            if(Auth::user()->role == 'employee'){
                $leaves = Leave::where('user_id',Auth::user()->id)->where('org_id', $org_id)->with(['user', 'leaveType'])->get();
            }else{
                $leaves = Leave::where('org_id', $org_id)->with(['user', 'leaveType'])->get();
            }
           

            return response()->json([
                'message' => 'Leaves fetched successfully',
                'leaves' => $leaves
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch leaves.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Add a new leave
    public function add_leave(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'leave_type_id' => 'required|exists:leave_types,id',
                'reason' => 'required|string',
                'start_date' => 'required|date',
                'end_date' => 'required|date|after_or_equal:start_date',
                'org_id' => 'required|exists:organizations,id',
                'status' => 'sometimes|string',
                'leave_status' => 'sometimes|string',
            ]);
    
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
    
            $leave = new Leave();
            $leave->user_id = $request->user_id;
            $leave->leave_type_id = $request->leave_type_id;
            $leave->reason = $request->reason;
            $leave->start_date = $request->start_date;
            $leave->end_date = $request->end_date;
            $leave->org_id = $request->org_id;
    
            // Set default values if not provided
            $leave->status = $request->status ?? 'pending'; // Default to 'pending'
            $leave->leave_status = $request->leave_status ?? 'processing'; // Default to 'processing'
    
            $leave->save();
    
            return response()->json([
                'success' => true,
                'message' => 'Leave created successfully',
                'leave' => $leave
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create leave.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    

    // Get a specific leave by ID
    public function get_leave($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $leave = Leave::where('org_id', $org_id)->with(['user', 'leaveType'])->findOrFail($id);

            return response()->json([
                'message' => 'Leave retrieved successfully',
                'leave' => $leave
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Leave not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve leave.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Update a specific leave
    public function update_leave(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'leave_type_id' => 'required|exists:leave_types,id',
                'reason' => 'string',
                'start_date' => 'string',
                'status' => 'string|in:approve,reject',
                'leave_status' => 'string|in:paid,unpaid,reject',
                'end_date' => 'string',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
            $leave = Leave::findOrFail($id);

            $leave->user_id = $request->user_id;
            $leave->leave_type_id = $request->leave_type_id;
            $leave->reason = $request->reason ?? $leave->reason;
            $leave->start_date = $request->start_date ?? $leave->start_date;
            $leave->end_date = $request->end_date ?? $leave->end_date;
            $leave->leave_status = $request->leave_status ?? $leave->leave_status;
            $leave->status = $request->status ?? $leave->status;
            $leave->save();

            return response()->json([
                'success' => true,
                'message' => 'Leave updated successfully',
                'leave' => $leave
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Leave not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update leave',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Delete a specific leave
    public function delete_leave($id)
    {
        try {
            $leave = Leave::findOrFail($id);
            $leave->delete();

            return response()->json(['message' => 'Leave deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Leave not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete leave',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_user_leaves($user_id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $user = User::with('userProfile.employees')->where('id', $user_id)->where('org_id', $org_id)->firstOrFail();
           
            if ($user->userProfile && $user->userProfile->employees->isNotEmpty()) {
                $totalAllocatedLeaves = $user->userProfile->employees->first()->total_leaves;
            } else {
                $totalAllocatedLeaves = 0;
            }

            $usedLeaves = Leave::where('user_id', $user_id)
                ->where('org_id', $org_id)
                ->where('status', 'approve')
                ->where('leave_status', 'paid')
                ->sum(DB::raw('DATEDIFF(end_date, start_date) + 1')); 

            // Calculate remaining leaves
            $remainingLeaves = $totalAllocatedLeaves - $usedLeaves;

            return response()->json([
                'success' => true,
                'message' => 'User leaves fetched successfully',
                'total_allocated_leaves' => (int) $totalAllocatedLeaves,
                'used_leaves' => (int) $usedLeaves,
                'remaining_leaves' => max(0, (int) $remainingLeaves) 
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User or leaves not found',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user leaves',
                'details' => $e->getMessage()
            ], 500);
        }
    }


}
