<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Organization;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use DataTables;
class OrganizationController extends Controller
{
    public function get_organizations_index()
    {
        try {
            // Fetch organizations with the associated user
            $organizations = Organization::with(['user'])
                ->where('id', '!=', Auth::user()->org_id) // Ensure we exclude the current user's organization
                ->get();

            return DataTables::of($organizations)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch organizations.',
                'details' => $e->getMessage(),
            ]);
        }
    }

    // Fetch all organizations
    public function get_organizations()
    {
        try {
            $organizations = Organization::with(['user'])->where('id', '!=', Auth::user()->id)->get();

            return response()->json([
                "message" => "Organizations fetched successfully",
                "organizations" => $organizations
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch organizations.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    // Add a new organization
    public function add_organization(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'name' => [
                    'required',
                    'string',
                    function ($attribute, $value, $fail) {
                        // Check if the name is unique within the entire organizations table
                        $exists = Organization::where('name', $value)->exists();
                        if ($exists) {
                            $fail('The name has already been taken in the organizations table.');
                        }
                    },
                ],
                'permissions' => 'required|array',
                'status' => 'nullable|string|in:active,inactive',
                'image' => 'nullable|file|max:2048',
            ]);


            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $organization = new Organization();
            $organization->user_id = $request->user_id;
            $organization->name = $request->name;
            $organization->permissions = $request->permissions;
            $organization->currency = $request->currency;
            $organization->status = $request->status ?? 'active';
            if ($request->hasFile('image')) {
                $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
                $imagePath = $request->file('image')->move(public_path('uploads/files/images'), $imageName);
                $organization->image = 'uploads/files/images/' . $imageName; // Save image path
            }
            $organization->save();

            User::find($request->user_id)->update(['org_id' => $organization->id, 'role' => 'organization']);

            return response()->json([
                'success' => true,
                'message' => 'Organization created successfully',
                'organization' => $organization
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create organization.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Fetch a single organization by ID
    public function get_organization($id)
    {
        try {
            $organization = Organization::with(['user'])->findOrFail($id);

            return response()->json([
                'message' => 'Organization retrieved successfully',
                'organization' => $organization
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Organization not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve organization.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Update an existing organization
    public function update_organization(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => [
                    'string',  // Name is optional, just ensure it's a string if provided
                    function ($attribute, $value, $fail) use ($id) {
                        if ($value) {
                            $exists = Organization::where('name', $value)
                                ->where('id', '!=', $id)
                                ->exists();
                            if ($exists) {
                                $fail('The name has already been taken in the organizations table.');
                            }
                        }
                    },
                ],
                'permissions' => 'nullable|array',  // Optional array for permissions
                'status' => 'nullable|string|in:active,inactive',  // Optional status
                'user_id' => 'nullable',  // Optional user_id if provided
                'image' => 'nullable|file|max:2048',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $organization = Organization::findOrFail($id);
            $oldUserId = $organization->user_id;

            if ($request->hasFile('image')) {
                $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
                $imagePath = $request->file('image')->move(public_path('uploads/files/images'), $imageName);
                $organization->image = 'uploads/files/images/' . $imageName; // Save new image path
            }

            // Update user role if user_id is changed
            if ($request->filled('user_id') && $oldUserId != $request->user_id) {
                $oldUser = User::find($oldUserId);
                if ($oldUser) {
                    $oldUser->role = 'organization';
                    $oldUser->save();
                }

                $newUser = User::find($request->user_id);
                if ($newUser) {
                    $newUser->role = 'organization';
                    $newUser->save();
                }

                $organization->user_id = $request->user_id;  // Update user_id
            }

            // Check if the status has changed
            $statusChanged = $request->has('status') && $organization->status !== $request->status;

            // Update organization details if provided
            if ($request->filled('name')) {
                $organization->name = $request->name;
            }
            if ($request->filled('currency')) {
                $organization->currency = $request->currency;
            }
            if ($request->filled('permissions')) {
                $organization->permissions = $request->permissions;
            }
            if ($request->filled('status')) {
                $organization->status = $request->status;
            }

            $organization->save();

            $user = User::find($organization->user_id);

            if ($request->filled('password')) {
                $user->password = bcrypt($request->password);
                $user->save();
            }

            if ($statusChanged) {
                if ($user) {
                    $user->status = $request->status;
                    $user->save();
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Organization updated successfully',
                'organization' => $organization
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Organization not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update organization',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    // Delete an organization
    public function delete_organization($id)
    {
        try {
            $organization = Organization::findOrFail($id);
            $organization->delete();

            return response()->json(['message' => 'Organization deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Organization not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete organization',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
