<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Loan;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class LoanController extends Controller
{
    public function get_loans_index()
    {
        try {
            $org_id = Auth::user()?->org_id;
            $loans = Loan::where('org_id', $org_id)
                ->with('employee.userProfile.user') 
                ->get();
    
            return DataTables::of($loans)
                ->addColumn('count', function($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->editColumn('employee_name', function($row) {
                    return $row->employee->userProfile->user->name ?? 'N/A';
                })
                ->editColumn('amount', function($row) {
                    return number_format($row['amount'], 2); 
                })
                ->editColumn('date', function($row) {
                    return \Carbon\Carbon::parse($row['date'])->format('d-M-Y'); 
                })
                ->editColumn('reason', function($row) {
                    return $row['reason']; 
                })
                ->editColumn('terms', function($row) {
                    return $row['terms']; 
                })
                ->editColumn('installment', function($row) {
                    return number_format($row['installment'], 2); 
                })
                ->editColumn('status', function($row) {
                    return $row['status'] ?? ''; 
                })
                
              
                ->rawColumns(['action'])
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function get_loans()
    {
        try {
            $org_id = Auth::user()->org_id;
            $loans = Loan::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->get()
            ->map(function ($increment) {
                return [
                    'id' => $increment->id,
                    'employee_id' => $increment->employee_id,
                    'amount' => $increment->amount,
                    'date' => $increment->date,
                    'reason' => $increment->reason,
                    'status' => $increment->status,
                    'terms' => $increment->terms,
                    'installment' => $increment->installment,
                    'employee_name' => $increment->employee->userProfile->user->name ?? 'N/A',
                ];
            });
            
            return response()->json([
                "message" => "Loans fetched successfully",
                "loans" => $loans
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch loans.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_loan(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'terms' => 'required|string',
                'installment' => 'required|numeric',
                'status' => 'required|string|in:active,inactive',
                'org_id' => 'required|exists:organizations,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $loan = new Loan();
            $loan->employee_id = $request->employee_id;
            $loan->amount = $request->amount;
            $loan->date = $request->date;
            $loan->org_id = $request->org_id;
            $loan->reason = $request->reason;
            $loan->terms = $request->terms;
            $loan->installment = $request->installment;
            $loan->status = $request->status;
            $loan->save();

            return response()->json([
                'success' => true,
                'message' => 'Loan created successfully',
                'loan' => $loan
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create loan.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_loan($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $loan = Loan::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->findOrFail($id)->get()
            ->map(function ($increment) {
                return [
                    'id' => $increment->id,
                    'employee_id' => $increment->employee_id,
                    'amount' => $increment->amount,
                    'date' => $increment->date,
                    'reason' => $increment->reason,
                    'status' => $increment->status,
                    'terms' => $increment->terms,
                    'installment' => $increment->installment,
                    'report' => $increment->report,
                    'employee_name' => $increment->employee->userProfile->user->name ?? 'N/A',
                ];
            });
            return response()->json([
                'message' => 'Loan retrieved successfully',
                'loan' => $loan
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Loan not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve loan.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_loan(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'terms' => 'required|string',
                'installment' => 'required|numeric',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $loan = Loan::findOrFail($id);

            $loan->employee_id = $request->employee_id;
            $loan->amount = $request->amount;
            $loan->date = $request->date;
            $loan->reason = $request->reason;
            $loan->terms = $request->terms;
            $loan->installment = $request->installment;
            $loan->status = $request->status;
            $loan->save();

            return response()->json([
                'success' => true,
                'message' => 'Loan updated successfully',
                'loan' => $loan
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Loan not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update loan',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_loan($id)
    {
        try {
            $loan = Loan::findOrFail($id);

            $loan->delete();

            return response()->json(['message' => 'Loan deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Loan not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete loan',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
