<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Settings\Loan;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Exception;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\Models\Settings\Payment;
use Carbon\Carbon;
use League\Csv\Writer;
use SplTempFileObject;
use Illuminate\Support\Facades\Auth;

class PaymentController extends Controller
{
    public function get_payments()
    {
        try {
            $org_id = Auth::user()->org_id;
            $payments = Payment::where('org_id', $org_id)->with(['employee.userProfile.user', 'organization'])
                ->get()
                ->map(function ($payment) {
                    return [
                        'id' => $payment->id,
                        'employee_name' => $payment->employee->userProfile->user->name ?? 'N/A',
                        'increments' => $payment->increments,
                        'deductions' => $payment->deductions,
                        'loan_amount' => $payment->loan_amount,
                        'loan_installment' => $payment->loan_installment,
                        'bonus_amount' => $payment->bonus_amount,
                        'net_pay' => $payment->net_pay,
                        'tax' => $payment->tax,
                        'net_pay_after_tax' => $payment->net_pay_after_tax,
                        'salary_month' => $payment->salary_month,
                    ];
                });

            return response()->json([
                "message" => "Payments fetched successfully",
                "payments" => $payments
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch payments.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_payment(Request $request)
    {

        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'org_id' => 'required|exists:organizations,id',
                'increments' => 'required',
                'deductions' => 'required',
                'allowances' => 'required',
                'loan_amount' => 'required',
                'unpaid_leaves' => 'required',
                'leave_deduction' => 'required',
                'loan_installment' => 'required',
                'loan_id' => 'nullable',
                'bonus_amount' => 'required',
                'net_pay' => 'required',
                'tax' => 'required',
                'net_pay_after_tax' => 'required',
                'salary_month' => 'required',
                'salary' => 'required',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $payment = new Payment();
            $payment->employee_id = $request->employee_id;
            $payment->org_id = $request->org_id;
            $payment->salary = $request->salary;
            $payment->salary_comment = $request?->salary_comment ?? "";
            $payment->increments = $request->increments;
            $payment->deductions = $request->deductions;
            $payment->allowances = $request->allowances;
            $payment->loan_amount = $request->loan_amount;
            $payment->leave_deduction = $request->leave_deduction;
            $payment->unpaid_leaves = $request->unpaid_leaves;
            $payment->loan_installment = $request->loan_installment;
            $payment->bonus_amount = $request->bonus_amount;
            $payment->net_pay = $request->net_pay;
            $payment->tax = $request->tax;
            $payment->net_pay_after_tax = $request->net_pay_after_tax;
            $payment->salary_month = $request->salary_month;
            $payment->save();

            if ($request?->loan_id != 0) {
                $remaining_amount = $request->loan_amount - $request->loan_installment;
                $loan = Loan::find($request->loan_id);
                
                if ($loan) {
                    $remaining_installment = $loan->installment - 1;
                    $status = $remaining_amount == 0 ? 'paid' : 'pending';
                    
                    $loan->update([
                        'report' => json_encode([
                            'loan_amount' => $request->loan_amount,
                            'paid_amount' => $request->loan_installment,
                            'remaining_amount' => $remaining_amount,
                            'remaining_installment' => $remaining_installment,
                            'pay_date' => Carbon::now(),
                        ]),
                        'status' => $status,
                    ]);
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Payment created successfully',
                'payment' => $payment
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_payment(Request $request, $id)
{
    try {
        // Validate request
        $validator = Validator::make($request->all(), [
            'month' => 'required',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        // Retrieve the first matching payment
        $payment = Payment::with([
            'employee.userProfile.user',
            'organization',
            'employee.department',
            'employee.designation',
            'employee.shift'
        ])
        ->where('employee_id', $id)
        ->where('salary_month', $request->month)
        ->first();

        if (!$payment) {
            throw new ModelNotFoundException('Payment not found');
        }

        $org_id = Auth::user()->org_id;
        // Convert total amount to words
        $totalAmountInWords = $this->convert_number_to_words($payment->net_pay_after_tax);

        // Prepare payslip data
        $payslipData = [
            'employee' => $payment->employee,
            'salary' => (int) $payment->salary,
            'increments' => (int) $payment->increments,
            'deductions' => (int) $payment->deductions,
            'allowances' => (int) $payment->allowances,
            'loan_amount' => (int) $payment->loan_amount,
            'loan_installment' => (int) $payment->loan_installment,
            'bonus_amount' => (int) $payment->bonus_amount,
            'unpaid_leaves' => (int) $payment->unpaid_leaves,
            'leave_deduction' => (int) $payment->leave_deduction,
            'net_pay' => (int) $payment->net_pay,
            'salary_comment' => $payment->salary_comment,
            'tax' => (int) ($payment->tax),
            'net_pay_after_tax' => (int) $payment->net_pay_after_tax,
            'salary_month' => $payment->salary_month,
            'payment_status' => $payment->exists() ? 'paid' : 'unpaid',
            'total_amount_in_words' => $totalAmountInWords,
        ];

        // Return successful response
        return response()->json([
            'message' => 'Payment retrieved successfully',
            'data' => $payslipData
        ], 200);
        
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'error' => 'Payment not found.',
            'details' => $e->getMessage()
        ], 404);
    } catch (Exception $e) {
        return response()->json([
            'error' => 'Failed to retrieve payment.',
            'details' => $e->getMessage()
        ], 500);
    }
}


    private function convert_number_to_words($number)
    {
        $hyphen = '-';
        $conjunction = ' and ';
        $separator = ', ';
        $negative = 'negative ';
        $decimal = ' point ';
        $dictionary = [
            0 => 'zero',
            1 => 'one',
            2 => 'two',
            3 => 'three',
            4 => 'four',
            5 => 'five',
            6 => 'six',
            7 => 'seven',
            8 => 'eight',
            9 => 'nine',
            10 => 'ten',
            11 => 'eleven',
            12 => 'twelve',
            13 => 'thirteen',
            14 => 'fourteen',
            15 => 'fifteen',
            16 => 'sixteen',
            17 => 'seventeen',
            18 => 'eighteen',
            19 => 'nineteen',
            20 => 'twenty',
            30 => 'thirty',
            40 => 'forty',
            50 => 'fifty',
            60 => 'sixty',
            70 => 'seventy',
            80 => 'eighty',
            90 => 'ninety',
            100 => 'hundred',
            1000 => 'thousand',
            1000000 => 'million',
            1000000000 => 'billion',
            1000000000000 => 'trillion',
        ];

        if (!is_numeric($number)) {
            return false;
        }

        if ($number < 0) {
            return $negative . $this->convert_number_to_words(abs($number));
        }

        $string = $fraction = null;

        if (strpos($number, '.') !== false) {
            list($number, $fraction) = explode('.', $number);
        }

        switch (true) {
            case $number < 21:
                $string = $dictionary[$number];
                break;
            case $number < 100:
                $tens = ((int) ($number / 10)) * 10;
                $units = $number % 10;
                $string = $dictionary[$tens];
                if ($units) {
                    $string .= $hyphen . $dictionary[$units];
                }
                break;
            case $number < 1000:
                $hundreds = $number / 100;
                $remainder = $number % 100;
                $string = $dictionary[$hundreds] . ' ' . $dictionary[100];
                if ($remainder) {
                    $string .= $conjunction . $this->convert_number_to_words($remainder);
                }
                break;
            default:
                $baseUnit = pow(1000, floor(log($number, 1000)));
                $numBaseUnits = (int) ($number / $baseUnit);
                $remainder = $number % $baseUnit;
                $string = $this->convert_number_to_words($numBaseUnits) . ' ' . $dictionary[$baseUnit];
                if ($remainder) {
                    $string .= $remainder < 100 ? $conjunction : $separator;
                    $string .= $this->convert_number_to_words($remainder);
                }
                break;
        }

        if (null !== $fraction && is_numeric($fraction)) {
            $string .= $decimal;
            $words = [];
            foreach (str_split((string) $fraction) as $number) {
                $words[] = $dictionary[$number];
            }
            $string .= implode(' ', $words);
        }

        return $string;
    }

    public function update_payment(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'increments' => 'required',
                'deductions' => 'required',
                'allowances' => 'required',
                'loan_amount' => 'required',
                'unpaid_leaves' => 'required',
                'leave_deduction' => 'required',
                'loan_installment' => 'required',
                'bonus_amount' => 'required',
                'net_pay' => 'required',
                'tax' => 'required',
                'net_pay_after_tax' => 'required',
                'salary_month' => 'required',
                'salary' => 'required',
                'salary_comment' => 'required',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $payment = Payment::findOrFail($id);
            $payment->employee_id = $request->employee_id;
            $payment->salary = $request->salary;
            $payment->salary_comment = $request->salary_comment;
            $payment->increments = $request->increments;
            $payment->deductions = $request->deductions;
            $payment->allowances = $request->allowances;
            $payment->loan_amount = $request->loan_amount;
            $payment->loan_installment = $request->loan_installment;
            $payment->bonus_amount = $request->bonus_amount;
            $payment->leave_deduction = $request->leave_deduction;
            $payment->unpaid_leaves = $request->unpaid_leaves;
            $payment->net_pay = $request->net_pay;
            $payment->tax = $request->tax;
            $payment->net_pay_after_tax = $request->net_pay_after_tax;
            $payment->salary_month = $request->salary_month;
            $payment->save();

            return response()->json([
                'success' => true,
                'message' => 'Payment updated successfully',
                'payment' => $payment
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Payment not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_payment($id)
    {
        try {
            $payment = Payment::findOrFail($id);
            $payment->delete();

            return response()->json(['message' => 'Payment deleted successfully'], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Payment not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete payment',
                'details' => $e->getMessage()
            ], 500);
        }
    }

   public function exportPayments(Request $request)
    {
        // Validate the request to ensure 'month' is provided
        $request->validate([
            'month' => 'required|date_format:Y-m'
        ]);
        $org_id = Auth::user()->org_id;
        // Fetch payments based on the salary_month field
        $payments = Payment::with(['employee.userProfile.user', 'organization'])->where('org_id', $org_id)->where('salary_month', $request->month)->get();
        // If no payments found, return an appropriate response
        if ($payments->isEmpty()) {
            return response()->json(['success' => false, 'message' => 'No payments found for the selected month'], 404);
        }
        // Create a CSV in memory
        // $csv = Writer::createFromFileObject(new SplTempFileObject());
        // // Add the header
        // $csv->insertOne([
        //     'organization',
        //     'email',
        //     'allowances',
        //     'increments',
        //     'deductions',
        //     'loan_amount',
        //     'loan_installment',
        //     'bonus_amount',
        //     'unpaid_leaves',
        //     'leave_deduction',
        //     'net_pay',
        //     'tax',
        //     'net_pay_after_tax',
        //     'days',
        //     'salary_month'
        // ]);
        // // Add the data rows
        // foreach ($payments as $payment) {
        //     $csv->insertOne([
        //         $payment->organization->name,
        //         $payment->employee->userProfile->user->email ?? 'N/A',
        //         $payment->allowances,
        //         $payment->increments,
        //         $payment->deductions,
        //         $payment->loan_amount,
        //         $payment->loan_installment,
        //         $payment->bonus_amount,
        //         $payment->unpaid_leaves,
        //         $payment->leave_deduction,
        //         $payment->net_pay,
        //         $payment->tax,
        //         $payment->net_pay_after_tax,
        //         $payment->days,
        //         $payment->salary_month
        //     ]);
        // }
        // // Store the CSV temporarily
        // $fileName = 'payments_' . $request->month . '.csv';
        // $filePath = 'temp/' . $fileName;
        // Storage::disk('local')->put($filePath, $csv->getContent());
        // // Generate a download link
        // $downloadLink = route('download', ['file' => $fileName]);
        // Return success response with download link
        return response()->json([
            'success' => true,
            'message' => 'Payments CSV generated successfully',
            'payments' => $payments
        ]);
    }

    public function download($file)
    {
        $filePath = 'temp/' . $file;

        if (Storage::disk('local')->exists($filePath)) {
            return Storage::download($filePath);
        }

        return response()->json(['success' => false, 'message' => 'File not found'], 404);
    }
}
