<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;

use Exception;
use App\Models\Settings\UserProfile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class UserProfileController extends Controller
{
    public function get_profiles()
    {
        try {
            $org_id = Auth::user()->org_id;
            $profiles = UserProfile::where('org_id',$org_id)->with('user')->get();
            return response()->json([
                "message" => "Profiles fetched successfully",
                "profiles" => $profiles
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch profiles.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_profile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'image' => 'required|image|mimes:jpeg,png,jpg|max:2048',
                'phone' => 'required|string|max:15',
                'address' => 'required|string',
                'cnic' => 'required',
                'date_of_birth' => 'required',
                'gender' => 'required|string|in:male,female,other',
                'user_id' => 'required|exists:users,id',
                'org_id' => 'required|exists:organizations,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Generate random names
            $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
            // $cnicName = Str::random(40) . '.' . $request->file('cnic')->getClientOriginalExtension();

            // Handle file uploads
            $imagePath = $request->file('image')->move(public_path('uploads/files/images'), $imageName);
            // $cnicPath = $request->file('cnic')->move(public_path('uploads/files/cnic'), $cnicName);

            $profile = new UserProfile();
            $profile->image = 'uploads/files/images/' . $imageName;
            $profile->phone = $request->phone;
            $profile->address = $request->address;
            // $profile->cnic = 'uploads/files/cnic/' . $cnicName;
            $profile->cnic = $request->cnic;
            $profile->date_of_birth = $request->date_of_birth;
            $profile->gender = $request->gender;
            $profile->user_id = $request->user_id;
            $profile->org_id = $request->org_id;
            $profile->save();

            return response()->json([
                'success' => true,
                'message' => 'Profile created successfully',
                'profile' => $profile
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create profile.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_profile($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $profile = UserProfile::where('org_id',$org_id)->with('user')->findOrFail($id);
            return response()->json([
                'message' => 'Profile retrieved successfully',
                'profile' => $profile
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Profile not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve profile.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_profile(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'phone' => 'required|string|max:15',
                'address' => 'required|string',
                'cnic' => 'required',
                'date_of_birth' => 'required',
                'gender' => 'required|string|in:male,female,other',
                'user_id' => 'required|exists:users,id'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $profile = UserProfile::findOrFail($id);

            // Handle the image file upload
            if ($request->hasFile('image')) {
                // Generate a unique file name
                $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
                $imagePath = public_path('uploads/files/images');

                // Delete old image if it exists
                if ($profile->image && file_exists(public_path($profile->image))) {
                    unlink(public_path($profile->image));
                }

                // Store the new image
                $request->file('image')->move($imagePath, $imageName);
                $profile->image = 'uploads/files/images/' . $imageName;
            }

            // Handle the CNIC file upload
            // if ($request->hasFile('cnic')) {
            //     // Generate a unique file name
            //     $cnicName = Str::random(40) . '.' . $request->file('cnic')->getClientOriginalExtension();
            //     $cnicPath = public_path('uploads/files/cnic');

            //     // Delete old CNIC file if it exists
            //     if ($profile->cnic && file_exists(public_path($profile->cnic))) {
            //         unlink(public_path($profile->cnic));
            //     }

            //     // Store the new CNIC
            //     $request->file('cnic')->move($cnicPath, $cnicName);
            //     $profile->cnic = 'uploads/files/cnic/' . $cnicName;
            // }

            // Update the remaining profile fields
            $profile->phone = $request->phone;
            $profile->address = $request->address;
            $profile->date_of_birth = $request->date_of_birth;
            $profile->gender = $request->gender;
            $profile->cnic = $request->cnic;
            $profile->user_id = $request->user_id;
            $profile->save();

            return response()->json([
                'success' => true,
                'message' => 'Profile updated successfully',
                'profile' => $profile
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        }
    }

    public function delete_profile($id)
    {
        try {
            $profile = UserProfile::findOrFail($id);

            // Delete associated files
            Storage::disk('public')->delete($profile->image);
            Storage::disk('public')->delete($profile->cnic);

            $profile->delete();

            return response()->json(['message' => 'Profile deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Profile not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete profile',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
