<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use League\Csv\Writer;
use SplTempFileObject;
use Illuminate\Http\Response;

class ReportController extends Controller
{
    public function finance_report(Request $request)
    {
        $month = $request->query('month');
        $year = $request->query('year');
        $type = $request->query('type', 'both');
        $startDate = $request->query('start_date');
        $endDate = $request->query('end_date');

        // Build the query string based on the inputs
        $queryParams = [];

        if ($month) {
            $queryParams['month'] = $month;
        } elseif ($year) {
            $queryParams['year'] = $year;
        } elseif ($startDate && $endDate) {
            $queryParams['start_date'] = $startDate;
            $queryParams['end_date'] = $endDate;
        }

        if ($type) {
            $queryParams['type'] = $type;
        }

        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/get/report/income?" . http_build_query($queryParams));
        // dd($response->json());
        $reportData = [];
        if ($response->successful()) {
            $reportData = $response->json()['report'] ?? [];

            // Merge incomes and expenses into a single array
            $mergedData = [];

            foreach ($reportData['incomes'] as $income) {
                $mergedData[] = [
                    'date' => $income['date'],
                    'description' => $income['description'],
                    'amount' => $income['amount'],
                    'type' => 'income',
                ];
            }

            foreach ($reportData['expenses'] as $expense) {
                $mergedData[] = [
                    'date' => $expense['date'],
                    'description' => $expense['description'],
                    'amount' => $expense['amount'],
                    'type' => 'expense',
                ];
            }

            // Sort merged data by date if needed
            usort($mergedData, function ($a, $b) {
                return strtotime($a['date']) - strtotime($b['date']);
            });

            $totals = [
                'total_income' => $reportData['total_income'] ?? 0,
                'total_expense' => $reportData['total_expense'] ?? 0,
                'total_revenue' => $reportData['total_revenue'] ?? 0,
                'type' => $reportData['type'] ?? 'both',
            ];
        } else {
            $mergedData = [];
            $totals = ['total_income' => 0, 'total_expense' => 0, 'total_revenue' => 0];
        }

        return view('admin.reports.finance_report', compact('mergedData', 'totals'));
    }




    public function employee_report(Request $request)
    {
        // Retrieve query parameters from the request
        $month = $request->query('month');
        $year = $request->query('year');
        $startDate = $request->query('start_date');
        $endDate = $request->query('end_date');

        $queryParams = [];

        if ($month) {
            // Search by year and month
            $queryParams['month'] = $month;
        } elseif ($year) {
            // Search by year
            $queryParams['year'] = $year;
        } elseif ($startDate && $endDate) {
            // Search by start and end date
            $queryParams['start_date'] = $startDate;
            $queryParams['end_date'] = $endDate;
        }

        // Get the bearer token from the session
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        // Make the API request with the constructed query parameters
        $response = Http::withToken($token)->get("{$apiBaseUrl}/get/report?" . http_build_query($queryParams));
        //    dd($response->json());
        // Initialize variables for report data and totals
        $reportData = [];
        $payments = [];
        $totals = [
            'total_net_pay' => 0,
            'total_increments' => 0,
            'total_deductions' => 0,
        ];

        // Check if the API response is successful
        if ($response->successful()) {
            // Get the report data from the response
            $reportData = $response->json()['report'] ?? [];

            // Extract the total values
            $totals['total_net_pay'] = $reportData['total_net_pay'] ?? 0;
            $totals['total_increments'] = $reportData['total_increments'] ?? 0;
            $totals['total_deductions'] = $reportData['total_deductions'] ?? 0;

            // Extract payments data
            $payments = $reportData['payments'] ?? [];
        }

        // Return the view with totals and payments data
        return view('admin.reports.employee_report', compact('totals', 'payments'));
    }

    public function employee_report_details($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/get/report/payment/{$id}");
            // dd($response->json());


            if ($response->successful()) {
                // Return the allowance data from the API response
                return response()->json($response->json()['payment']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Payment not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    public function export_payments(Request $request)
    {
        $validatedData = $request->validate([
            'month' => 'required|date_format:Y-m',
        ]);
    
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
    
        $response = Http::withToken($token)->get("{$apiBaseUrl}/export-payments?month={$validatedData['month']}");
    
        if ($response->successful()) {
            $payments = $response->json()['payments'];
    
            $csv = Writer::createFromFileObject(new SplTempFileObject());
            // Add the header
            $csv->insertOne([
                'organization',
                'email',
                'allowances',
                'increments',
                'deductions',
                'loan_amount',
                'loan_installment',
                'bonus_amount',
                'unpaid_leaves',
                'leave_deduction',
                'net_pay',
                'tax',
                'net_pay_after_tax',
                'days',
                'salary_month'
            ]);
    
            // Add the data rows
            foreach ($payments as $payment) {
                $csv->insertOne([
                    $payment['organization']['name'], // Correctly access organization name
                    $payment['employee']['user_profile']['user']['email'] ?? 'N/A', // Correctly access email
                    $payment['allowances'],
                    $payment['increments'],
                    $payment['deductions'],
                    $payment['loan_amount'],
                    $payment['loan_installment'],
                    $payment['bonus_amount'],
                    $payment['unpaid_leaves'],
                    $payment['leave_deduction'],
                    $payment['net_pay'],
                    $payment['tax'],
                    $payment['net_pay_after_tax'],
                    $payment['days'],
                    $payment['salary_month']
                ]);
            }
    
            // Create a response for download
            return response((string) $csv)
                ->header('Content-Type', 'text/csv')
                ->header('Content-Disposition', 'attachment; filename="payments.csv"');
        }
    
        // Handle error response
        return response()->json(['error' => 'Unable to fetch employee data'], Response::HTTP_BAD_REQUEST);
    }
    
}
