<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\DataTables\UserDataTable;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;

class UserController extends Controller
{
    private function fetch_employees(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");

        if ($response->successful()) {
            // Fetch the employee data including related user name
            $employees = $response->json()['employees'] ?? [];

            // Extract only the employee ID and the associated user name
            $employees = array_map(function ($employee) {
                return [
                    'id' => $employee['user_profile']['user']['id'] ?? null, // Employee ID
                    'name' => $employee['user_profile']['user']['name'] ?? null, // Associated User's Name
                ];
            }, $employees);

            return $employees;
        }

        return [];
    }
    public function user_list(UserDataTable $dataTable)
    {
        $employees = $this->fetch_employees();
        return $dataTable->render('admin.user.user',compact('employees'));
    }
    public function add_user_form(Request $request)
    {
        // dd($request->permissions);
        // Validate incoming request
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email', // Ensure unique email
            'password' => 'required|string|min:8|confirmed',
            'permissions' => 'required|array|min:1',
            'password_confirmation'=> 'required' // Ensure at least one permission is provided
        ]);
    
        try {
            // Get the bearer token and organization ID from the session
            $token = session('bearer_token');
            $org_id = session('org_id');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Prepare the request data
            $requestData = [
                'name' => $validatedData['name'],
                'email' =>$validatedData['email'],
                'permissions' => $validatedData['permissions'],
                'password' => $validatedData['password'],
                'password_confirmation' => $validatedData['password_confirmation'],
                'org_id' => $org_id,
                'role' => 'normal'
            ];
   
            $response = Http::withToken($token)->post("{$apiBaseUrl}/users/store", $requestData);
            // dd($response->json());
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'User added successfully',
                ], 201);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'User addition failed',
                    'errors' => $error['errors'] ?? [] // Handle detailed error messages if available
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle any unexpected exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage(),
            ], 500);
        }
    }
    
    
    

    public function update_user_form(Request $request, $id)
    {
        // Validate incoming request
        $request->validate([
            'name' => 'nullable|string|max:255',
            'email' => 'nullable|string|email|max:255',
            'password' => 'nullable|string|min:8|confirmed',
            'permissions' => 'nullable|array|min:1',  
        ]);
    
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send a PUT request to update the user information
            $response = Http::withToken($token)->put("{$apiBaseUrl}/users/update/{$id}", [
                'name' => $request->name,
                'email' => $request->email,
                'password' => $request->password,
                'password_confirmation' => $request->password_confirmation,
                'permissions' => $request->permissions,
            ]);
    
            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'User updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update user.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    


    public function delete_user($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/users/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'User deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete User'
            ], $response->status());
        }
    }

    public function get_user($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/users/get/{$id}");



            if ($response->successful()) {
                // Return the allowance data from the API response
                return response()->json($response->json()['user']);
            } else {
                return response()->json(['status' => 404, 'message' => 'User not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    public function update_user_status(Request $request)
    {
        try {
            // Validate incoming request
            $request->validate([
                'id' => 'required|integer', // Ensure 'id' is present and is an integer
            ]);
    
            // Get the bearer token from the session and the base API URL from environment config
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            
            // Prepare the payload with table and ID
            $payload = [
                'table' => 'users',  // The table name is hardcoded as 'organizations'
                'id' => $request->id  // ID is passed from the request
            ];
            
            // Send a POST request to the external API to update the status
            $response = Http::withToken($token)->post("{$apiBaseUrl}/update/status", $payload);
    
            // Check if the response was successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Status updated successfully'
                ]);
            } else {
                // If the request fails, return the error message from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update status'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Catch any exceptions and return a 500 response with the error message
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
}
