<?php

namespace App\DataTables\Settings;

use Yajra\DataTables\DataTableAbstract;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Illuminate\Support\Facades\Http;

class DocumentDataTable extends DataTable
{
    protected $imageBaseUrl;

    public function __construct()
    {
        // Initialize the base URL for images
        $this->imageBaseUrl = env('API_URL_FOR_IMAGE');
    }

    public function dataTable($query): DataTableAbstract
    {
        return datatables()->of($query)
            ->editColumn('document', function ($document) {
                // Generate the full URL for the document
                $fullUrl = $this->imageBaseUrl . '/' . $document['document'];
    
                // Return an HTML image tag with the URL
                return '<img src="' . $fullUrl . '" alt="Document Image" style="width: 50px; height: 50px;">';
            })
            ->rawColumns(['document']); // Allow HTML rendering in the 'document' column
    }
    

    /**
     * Get the query source of dataTable.
     *
     * @return \Illuminate\Support\Collection
     */
    public function query()
    {
        // Fetch data from the external API
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("$apiBaseUrl/documents");
      
        // Check if the response is successful
        if ($response->successful()) {
            // Return the JSON data from 'documents' key
            return collect($response->json()['documents']);
        }

        // Return an empty collection if the API call fails
        return collect([]);
    }

    /**
     * Optional method if you want to use the HTML builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('documents_table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    ->orderBy(1)
                    ->selectStyleSingle();
    }

    /**
     * Get the DataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('id'),
            Column::make('document')->title('Document'), // Custom column title
            Column::make('name'),
            Column::make('status'),
            Column::make('remind_date'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Document_' . date('YmdHis');
    }
}
