<?php

namespace App\Http\Controllers\SuperAdmin;

use Illuminate\Http\Request;
use App\DataTables\UserDataTable;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;

class AdminController extends Controller
{
    
    public function admin_list(UserDataTable $dataTable)
    {
        return $dataTable->render('super_admin.manage_users.manage_users');
    }
    public function add_admin_form(Request $request)
    {
        // Validate incoming request
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'password' => 'required|string|min:8|confirmed', // Password confirmation must match
        ]);

        try {
            // Get the bearer token from the session
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send POST request to the external API
            $response = Http::withToken($token)->post("{$apiBaseUrl}/users/store", [
                'name' => $request->name,
                'email' => $request->email,
                'password' => $request->password,
                'password_confirmation' => $request->password_confirmation, // Ensure confirmation is passed
                'role'=>$request->role
            ]);

            // Check if the response is successful
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'User added successfully',
                ], 201);
            } else {
                // Handle errors from the API response
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add User.',
                    'errors' => $error['errors'] ?? [] // Handle detailed error messages if available
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle any unexpected exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function update_admin_form(Request $request, $id)
    {
        // Validate incoming request
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'password' => 'required|string|min:8|confirmed', // Password confirmation must match
        ]);
    
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send a PUT request to update the user information
            $response = Http::withToken($token)->put("{$apiBaseUrl}/users/update/{$id}", [
                'name' => $request->name,
                'email' => $request->email,
                'password' => $request->password,
                'password_confirmation' => $request->password_confirmation,
                'role'=>$request->role
            ]);
    
            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'User updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update user.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    


    public function delete_admin($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/users/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'User deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete User'
            ], $response->status());
        }
    }

    public function get_admin($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/users/get/{$id}");



            if ($response->successful()) {
                // Return the allowance data from the API response
                return response()->json($response->json()['user']);
            } else {
                return response()->json(['status' => 404, 'message' => 'User not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    
    
}
